// We use babel plugin macros here to work with CRA. Normally you would use
// `import { graphql } from 'react-relay';` with a proper babel config.
// @ts-expect-error: no type-defs available
import graphql from 'babel-plugin-relay/macro';
import {
  PlayerType,
  createStreamManifestUrl,
  useStreamAnalytics,
} from 'pulsar';
import type { PulsarProps } from 'pulsar';
import type { FC } from 'react';
import { useFragment } from 'react-relay/hooks';
import { useStaticEnvironment } from 'tachyon-environment';
import { exhaustedCase } from 'tachyon-utils';
import {
  PulsarCoreBackend,
  PulsarHlsjsBackend,
  PulsarNativeBackend,
} from '../../common/PulsarConsumer';
import { usePlayerConsumer } from '../../common/PulsarConsumer/PulsarConsumerRoot';
import type { CommonPageProps } from '../../types';
import type { StreamPlayer_channel$key } from './__generated__/StreamPlayer_channel.graphql';

const channelFragment = graphql`
  fragment StreamPlayer_channel on User {
    id
    login
    roles {
      isPartner
    }
    self {
      subscriptionBenefit {
        id
      }
    }
    stream {
      id
      game {
        name
      }
      playbackAccessToken(
        params: {
          playerType: "pulsar"
          playerBackend: "mediaplayer"
          platform: "web"
        }
      ) {
        signature
        value
      }
      previewImageURL
    }
  }
`;

export type StreamPlayerProps = Partial<PulsarProps> & {
  channel: StreamPlayer_channel$key;
  forceServerPreRoll?: boolean | undefined;
  playerBackend: CommonPageProps['playerBackend'];
};

export const StreamPlayer: FC<StreamPlayerProps> = ({
  channel: channelRef,
  forceServerPreRoll,
  playerBackend,
  ...pulsarProps
}) => {
  const channelData = useFragment(channelFragment, channelRef);

  const { markEvent } = usePlayerConsumer();
  const {
    client: { sessionID },
  } = useStaticEnvironment();

  useStreamAnalytics({
    channel: channelData,
    currentUser: null,
    onTrackingEvent: markEvent,
    tracking: {
      playerType: PlayerType.Pulsar,
      sessionId: sessionID,
    },
  });

  const manifestUrl = createStreamManifestUrl(channelData, {
    forceServerPreRoll,
  });
  if (!manifestUrl) {
    throw new Error('Unable to create Stream manifest URL');
  }

  switch (playerBackend) {
    case 'mediaplayer':
      return <PulsarCoreBackend src={manifestUrl} {...pulsarProps} />;
    case 'native':
      return <PulsarNativeBackend src={manifestUrl} {...pulsarProps} />;
    case 'hlsjs':
      return <PulsarHlsjsBackend src={manifestUrl} {...pulsarProps} />;
    default:
      return exhaustedCase(playerBackend, null);
  }
};
