// We use babel plugin macros here to work with CRA. Normally you would use
// `import { graphql } from 'react-relay';` with a proper babel config.
// @ts-expect-error: no type-defs available
import graphql from 'babel-plugin-relay/macro';
import type { FC } from 'react';
import { useState } from 'react';
import { useLazyLoadQuery } from 'react-relay/hooks';
import { CheckBox, Display, FlexDirection, Layout } from 'twitch-core-ui';
import { PulsarConsumer } from '../../common/PulsarConsumer';
import type { CommonPageProps } from '../../types';
import { StreamPlayer } from './StreamPlayer';
import { StreamSetter } from './StreamSetter';
import type { StreamPage_Query } from './__generated__/StreamPage_Query.graphql';

// Consistent fragment for channel
// eslint-disable-next-line no-unused-expressions
graphql`
  fragment StreamPage_channel on User {
    login
    stream {
      id
      game {
        name
      }
      playbackAccessToken(
        params: {
          playerType: "pulsar"
          playerBackend: "mediaplayer"
          platform: "web"
        }
      ) {
        signature
        value
      }
      previewImageURL
    }
  }
`;

const streamPageQuery = graphql`
  query StreamPage_Query($login: String!, $noChannel: Boolean!) {
    topStream: streams(first: 1) @include(if: $noChannel) {
      edges {
        node {
          broadcaster {
            ...StreamPage_channel @relay(mask: false)
            ...StreamPlayer_channel
          }
        }
      }
    }
    channel: user(login: $login) @skip(if: $noChannel) {
      ...StreamPage_channel @relay(mask: false)
      ...StreamPlayer_channel
    }
  }
`;

type StreamPageProps = CommonPageProps;

export const StreamPage: FC<StreamPageProps> = ({
  appInitTimeMS,
  playerBackend,
}) => {
  const [selectedChannel, setSelectedChannel] = useState<string | null>(null);
  const [autoPlay, setAutoPlay] = useState(true);
  const [forceServerPreRoll, setForceServerPreRoll] = useState(false);

  const data = useLazyLoadQuery<StreamPage_Query>(streamPageQuery, {
    login: selectedChannel ?? '',
    noChannel: selectedChannel === null,
  });

  const channelData =
    data?.channel ?? data?.topStream?.edges?.[0]?.node?.broadcaster ?? null;
  if (!channelData) {
    throw new Error('No Channel Data :(');
  }

  return (
    <div>
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        padding={{ left: 1 }}
      >
        <StreamSetter
          defaultStreamLogin={channelData?.login ?? ''}
          onStreamSet={setSelectedChannel}
        />
        <CheckBox
          checked={autoPlay}
          label="Autoplay"
          onChange={(event) => {
            setAutoPlay(event.currentTarget.checked);
          }}
        />
        <CheckBox
          checked={forceServerPreRoll}
          label="Force Server Pre-Roll"
          onChange={(event) => {
            const enable = event.currentTarget.checked;
            if (enable) {
              window.alert(
                'Requires Staff account or Amazon VPN and a channel with pre-rolls enabled',
              );
            }
            setForceServerPreRoll(enable);
          }}
        />
      </Layout>
      <PulsarConsumer
        appInitTimeMS={appInitTimeMS}
        captionContentKey={`stream-${channelData.stream?.id}`}
        displayVideoTime={false}
      >
        <StreamPlayer
          autoPlay={autoPlay}
          channel={channelData}
          forceServerPreRoll={forceServerPreRoll}
          playerBackend={playerBackend}
        />
      </PulsarConsumer>
    </div>
  );
};
