// We use babel plugin macros here to work with CRA. Normally you would use
// `import { graphql } from 'react-relay';` with a proper babel config.
// @ts-expect-error: no type-defs available
import graphql from 'babel-plugin-relay/macro';
import { PlayerType, createVideoManifestUrl, useVodAnalytics } from 'pulsar';
import type { PulsarProps } from 'pulsar';
import type { FC } from 'react';
import { useFragment } from 'react-relay/hooks';
import { useStaticEnvironment } from 'tachyon-environment';
import { convertToUnsafeID } from 'tachyon-relay';
import { exhaustedCase } from 'tachyon-utils';
import {
  PulsarCoreBackend,
  PulsarHlsjsBackend,
  PulsarNativeBackend,
} from '../../common/PulsarConsumer';
import { usePlayerConsumer } from '../../common/PulsarConsumer/PulsarConsumerRoot';
import type { CommonPageProps } from '../../types';
import type { VideoPlayer_video$key } from './__generated__/VideoPlayer_video.graphql';

const videoPlayerFragment = graphql`
  fragment VideoPlayer_video on Video {
    id
    broadcastType
    game {
      name
    }
    owner {
      id
      login
      roles {
        isPartner
      }
      self {
        subscriptionBenefit {
          id
        }
      }
    }
    playbackAccessToken(
      params: {
        playerType: "pulsar"
        playerBackend: "mediaplayer"
        platform: "web"
      }
    ) {
      signature
      value
    }
    previewThumbnailURL
  }
`;

export type VideoPlayerProps = Partial<PulsarProps> & {
  playerBackend: CommonPageProps['playerBackend'];
  video: VideoPlayer_video$key;
};

export const VideoPlayer: FC<VideoPlayerProps> = ({
  playerBackend,
  video: videoRef,
  ...pulsarProps
}) => {
  const videoData = useFragment(videoPlayerFragment, videoRef);

  const { markEvent } = usePlayerConsumer();
  const {
    client: { sessionID },
  } = useStaticEnvironment();

  useVodAnalytics({
    currentUser: null,
    onTrackingEvent: markEvent,
    tracking: {
      playerType: PlayerType.Pulsar,
      sessionId: sessionID,
    },
    video: videoData,
  });

  const manifestUrl = createVideoManifestUrl({
    id: convertToUnsafeID(videoData.id),
    playbackAccessToken: videoData.playbackAccessToken,
  });
  if (!manifestUrl) {
    throw new Error('Unable to create VOD manifest URL');
  }

  switch (playerBackend) {
    case 'mediaplayer':
      return <PulsarCoreBackend src={manifestUrl} {...pulsarProps} />;
    case 'native':
      return <PulsarNativeBackend src={manifestUrl} {...pulsarProps} />;
    case 'hlsjs':
      return <PulsarHlsjsBackend src={manifestUrl} {...pulsarProps} />;
    default:
      return exhaustedCase(playerBackend, null);
  }
};
