import { useState } from 'react';
import type { ChangeEvent, FC, MouseEvent } from 'react';
import {
  AlignItems,
  Button,
  CoreText,
  Display,
  Input,
  InputType,
  Layout,
} from 'twitch-core-ui';

export type VideoSet = {
  id: string;
  startFromTime: number;
};

type VideoSetterProps = {
  defaultVideoId: string;
  onVideoSet: (videoSet: VideoSet) => void;
};

export const VideoSetter: FC<VideoSetterProps> = ({
  defaultVideoId,
  onVideoSet,
}) => {
  const [videoId, setVideoId] = useState(defaultVideoId);
  const [startFromTime, setStartFromTime] = useState(0);

  function inputChangeHandler(e: ChangeEvent<HTMLInputElement>): void {
    setVideoId(e.target.value.toLowerCase());
  }

  function startFromTimeChangeHandler(e: ChangeEvent<HTMLInputElement>): void {
    setStartFromTime(Number.parseInt(e.target.value, 10));
  }

  function playClickHandler(event: MouseEvent<HTMLElement>): void {
    event.preventDefault();
    onVideoSet({
      id: videoId,
      startFromTime,
    });
  }

  return (
    <Layout alignItems={AlignItems.Center} display={Display.Flex}>
      <CoreText>Video ID</CoreText>
      <Layout margin={{ x: 1 }}>
        <Input
          defaultValue={videoId}
          onChange={inputChangeHandler}
          type={InputType.Text}
        />
      </Layout>
      <CoreText>Start Time</CoreText>
      <Layout padding={{ x: 1 }}>
        <Input
          defaultValue={'0'}
          onChange={startFromTimeChangeHandler}
          type={InputType.Number}
        />
      </Layout>
      <Button
        disabled={
          !videoId ||
          typeof startFromTime === 'undefined' ||
          videoId === defaultVideoId
        }
        onClick={playClickHandler}
      >
        Load Video
      </Button>
    </Layout>
  );
};
