// We use babel plugin macros here to work with CRA. Normally you would use
// `import { graphql } from 'react-relay';` with a proper babel config.
// @ts-expect-error: no type-defs available
import graphql from 'babel-plugin-relay/macro';
import type { FC } from 'react';
import { useState } from 'react';
import { useLazyLoadQuery } from 'react-relay/hooks';
import { convertToUnsafeID } from 'tachyon-relay';
import { AlignItems, CheckBox, Display, Layout } from 'twitch-core-ui';
import { PulsarConsumer } from '../../common/PulsarConsumer';
import type { CommonPageProps } from '../../types';
import { VideoPlayer } from './VideoPlayer';
import { VideoSetter } from './VideoSetter';
import type { VideoPage_Query } from './__generated__/VideoPage_Query.graphql';

// Consistent fragment for video
// eslint-disable-next-line no-unused-expressions
graphql`
  fragment VideoPage_video on Video {
    id
  }
`;

const videoPageQuery = graphql`
  query VideoPage_Query($videoId: ID!, $noVideo: Boolean!) {
    topVideo: videos(first: 1) @include(if: $noVideo) {
      edges {
        node {
          ...VideoPage_video @relay(mask: false)
          ...VideoPlayer_video
        }
      }
    }
    video(id: $videoId) @skip(if: $noVideo) {
      ...VideoPage_video @relay(mask: false)
      ...VideoPlayer_video
    }
  }
`;

type VideoPageProps = CommonPageProps;

export const VideoPage: FC<VideoPageProps> = ({
  appInitTimeMS,
  playerBackend,
}) => {
  const [selectedVideoId, setSelectedVideoId] = useState<string | null>(null);
  const [autoPlay, setAutoPlay] = useState(true);
  const [startFromTime, setStartFromTime] = useState<number | undefined>(
    undefined,
  );

  const data = useLazyLoadQuery<VideoPage_Query>(videoPageQuery, {
    noVideo: selectedVideoId === null,
    videoId: selectedVideoId ?? '',
  });

  const videoData = data?.video ?? data?.topVideo?.edges?.[0]?.node ?? null;
  if (!videoData) {
    throw new Error('No video data :(');
  }

  return (
    <>
      <Layout alignItems={AlignItems.Center} display={Display.Flex} padding={2}>
        <VideoSetter
          defaultVideoId={convertToUnsafeID(videoData?.id) ?? ''}
          onVideoSet={({ id, startFromTime: newStartFromTime }) => {
            setSelectedVideoId(id);
            setStartFromTime(newStartFromTime);
          }}
        />
        <Layout padding={{ left: 1 }}>
          <CheckBox
            checked={autoPlay}
            label="Autoplay"
            onChange={(event) => {
              setAutoPlay(event.currentTarget.checked);
            }}
          />
        </Layout>
      </Layout>
      <PulsarConsumer
        appInitTimeMS={appInitTimeMS}
        captionContentKey={`video-${videoData.id}`}
        displayVideoTime
      >
        <VideoPlayer
          autoPlay={autoPlay}
          playerBackend={playerBackend}
          startTime={startFromTime}
          video={videoData}
        />
      </PulsarConsumer>
    </>
  );
};
