import type { FC } from 'react';
import { useCallback, useEffect, useRef, useState } from 'react';
import ReactDOM from 'react-dom';
import styled from 'styled-components';
import { CoreUiSsrRoot } from 'tachyon-more-ui';
import {
  ConfirmationScreen,
  ContentSelect,
  DeadEndScreen,
  DescriptionForm,
  DetailedReasonSelect,
  FormPreview,
  NetzDGForm,
  ReasonSelect,
  ReportWizardRoot,
} from 'tachyon-report-wizard';
import { useStorage } from 'tachyon-utils';
import {
  AlertBanner,
  AlertBannerType,
  AlignItems,
  Background,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  Position,
  ThemeToggle,
  Tooltip,
} from 'twitch-core-ui';
import {
  sampleDetailedReasons,
  sampleNetzDGReasons,
  sampleReportWizardNetzDGData,
  sampleReportWizardTOSData,
  sampleReportableContent,
} from './data';
import { WizardExample } from './wizard-example';

// ENABLE_NETZDG_DATA flag to use NetzDG data. ReportWizardRoot validates
// data and is optimized via useConst so the wizard does not support dynamic
// data changes as data in actual use case is not expected to be changing.
const ENABLE_NETZDG_DATA = false;

interface ComponentDemo {
  examples: JSX.Element[];
  name: string;
}

const ScExampleContainer = styled.div`
  width: 100vw;
`;

export const Example: FC = () => {
  const [theme, setTheme] = useStorage<'dark' | 'light'>(
    'report-wizard-demo-theme',
    'light',
  );

  const currentTimeout = useRef<ReturnType<typeof setTimeout>>();
  const [toastDisplay, setToastDisplay] = useState<{
    message?: string;
    status: string;
  }>();
  const [isLoading, setIsLoading] = useState(false);
  const [submittedReport, setSubmittedReport] = useState(false);

  const handleOnNetzDGFormValidationChange = useCallback(
    (formValues) =>
      setToastDisplay({
        message: JSON.stringify(formValues),
        status: 'NetzDG form validation change: ',
      }),
    [],
  );

  useEffect(() => {
    if (currentTimeout.current) {
      clearTimeout(currentTimeout.current);
    }

    if (toastDisplay) {
      currentTimeout.current = setTimeout(() => {
        setToastDisplay(undefined);
      }, 4000);
    }

    return () => {
      if (currentTimeout.current) {
        clearTimeout(currentTimeout.current);
      }
    };
  }, [toastDisplay]);

  const components: ComponentDemo[] = [
    {
      examples: [
        <FormPreview
          content="CHAT_REPORT"
          detailedReason={sampleDetailedReasons[0]}
          key="0"
          reason={sampleNetzDGReasons[0]}
          username="username"
        />,
      ],
      name: 'FormPreview',
    },
    {
      examples: [
        <ContentSelect
          contents={sampleReportableContent}
          key="0"
          onSelect={(content) =>
            setToastDisplay({ message: content, status: 'Selected Content: ' })
          }
        />,
      ],
      name: 'ContentSelect',
    },
    {
      examples: [
        <DeadEndScreen deadEndType={'LINK_DMCA_GUIDELINES'} key="0" />,
      ],
      name: 'DeadEnd',
    },
    {
      examples: [
        <ReasonSelect
          allReasons={sampleNetzDGReasons}
          key="0"
          mainReasons={sampleNetzDGReasons.slice(0, 4)}
          onExitSearchOther={() =>
            setToastDisplay({ status: 'Handling onExitSearchOther' })
          }
          onSearchOther={() =>
            setToastDisplay({ status: 'Handling onSearchOther' })
          }
          onSelect={(reason, detailedReason) =>
            setToastDisplay({
              message: `${reason}, ${detailedReason}`,
              status: 'Selected Reason and DetailedReason: ',
            })
          }
        />,
      ],
      name: 'ReasonSelect',
    },
    {
      examples: [
        <DetailedReasonSelect
          key="1"
          onSelect={(reason) =>
            setToastDisplay({
              message: reason,
              status: 'Selected Detailed Reason: ',
            })
          }
          reasons={sampleDetailedReasons}
        />,
      ],
      name: 'DetailedReasonSelect',
    },
    {
      examples: [
        <DescriptionForm
          key="1"
          onChange={(description) =>
            setToastDisplay({
              message: description,
              status: 'Description: ',
            })
          }
        />,
      ],
      name: 'DescriptionForm',
    },
    {
      examples: [
        <NetzDGForm
          key="1"
          onFormValidationChange={handleOnNetzDGFormValidationChange}
        />,
      ],
      name: 'NetzDGForm',
    },
    {
      examples: [
        <ConfirmationScreen
          key="0"
          onClickClose={() =>
            setToastDisplay({ status: 'Handling onClickClose' })
          }
          onToggleBlockUser={(isBlocked) =>
            setToastDisplay({
              message: `${isBlocked}`,
              status: 'Handling onToggleBlockUser: ',
            })
          }
        />,
      ],
      name: 'ConfirmationScreen',
    },
  ];

  return (
    <CoreUiSsrRoot appRootElementId="root" theme={theme}>
      <ScExampleContainer>
        <Layout
          background={Background.Base}
          color={Color.Base}
          fullHeight
          fullWidth
          padding={5}
          position={Position.Relative}
        >
          {toastDisplay && (
            <Layout attachLeft attachRight attachTop position={Position.Fixed}>
              <AlertBanner
                closeButton={{
                  'aria-label': 'Close Alert',
                  onClick: () => setToastDisplay(undefined),
                }}
                message={toastDisplay.message || ''}
                status={toastDisplay.status}
                type={AlertBannerType.Info}
              />
            </Layout>
          )}

          <Layout>
            <Layout
              alignItems={AlignItems.Center}
              display={Display.Flex}
              fullWidth
              margin={{ bottom: 1 }}
            >
              <CoreText fontSize={FontSize.Size3}>Report Wizard Flow</CoreText>
              <Layout margin={{ left: 'auto', right: 0.5 }}>
                <Tooltip label="To enable NetzDG, modify const ENABLE_NETZDG_DATA">
                  <CoreText>{`NetzDG data ${
                    ENABLE_NETZDG_DATA ? 'enabled' : 'disabled'
                  }`}</CoreText>
                </Tooltip>
              </Layout>

              <ThemeToggle
                checked={theme === 'dark'}
                onChange={(e) =>
                  setTheme(e.currentTarget.checked ? 'dark' : 'light')
                }
              />
            </Layout>

            <ReportWizardRoot
              contentID="contentID"
              data={
                ENABLE_NETZDG_DATA
                  ? sampleReportWizardNetzDGData
                  : sampleReportWizardTOSData
              }
              fromUserID="123"
              hideBlockPrompt={false}
              onCloseWizard={() =>
                setToastDisplay({ status: 'Report Wizard onCloseWizard' })
              }
              onSubmit={(form) => {
                setIsLoading(true);
                setToastDisplay({
                  message: JSON.stringify(form),
                  status: 'Report Wizard onSubmit: ',
                });
                setTimeout(() => {
                  setIsLoading(false);
                  setSubmittedReport(true);
                }, 2000);
              }}
              onToggleBlock={(block) => {
                setToastDisplay({
                  message: `${block}`,
                  status: 'Report Wizard onToggleBlock: ',
                });
              }}
              reportSubmissionLoading={isLoading}
              reportSubmissionSuccess={submittedReport}
              targetUserID="321"
              targetUserUsername="targetUserUsername"
            >
              <WizardExample />
            </ReportWizardRoot>
          </Layout>
          <Layout
            borderBottom
            display={Display.Flex}
            flexDirection={FlexDirection.Row}
            fullWidth
            justifyContent={JustifyContent.Between}
            margin={{ bottom: 2 }}
            padding={{ bottom: 1 }}
          >
            <CoreText fontSize={FontSize.Size3}>
              Report Wizard Components
            </CoreText>
          </Layout>

          {components.map(({ examples, name }) => (
            <Layout
              borderBottom
              key={name}
              margin={{ bottom: 1 }}
              padding={{ bottom: 2 }}
            >
              <CoreText fontSize={FontSize.Size4}>{`<${name} />`}</CoreText>
              <Layout
                display={Display.Flex}
                flexDirection={FlexDirection.Row}
                justifyContent={JustifyContent.Between}
                padding={{ top: 1 }}
              >
                {examples}
              </Layout>
            </Layout>
          ))}
        </Layout>
      </ScExampleContainer>
    </CoreUiSsrRoot>
  );
};

Example.displayName = 'Example';

ReactDOM.render(<Example />, document.getElementById('root'));
