import type { FC } from 'react';
import { useState } from 'react';
import { NodeNav, VerticalNav, useFocus } from 'tachyon-tv-nav';
import {
  Display,
  FlexDirection,
  Layout,
  SegmentedButton,
  SegmentedButtonOption,
} from 'twitch-core-ui';
import { SimpleShelf } from '../common';

const Leaf: FC<{ focusIndex: number }> = ({ focusIndex }) => {
  const { focusId, focused } = useFocus(focusIndex);
  return (
    <div>
      {focusId}:{focused ? 'Focused!' : 'Not Focused!'}
    </div>
  );
};

type DynamicPageProps = {
  focusIndex: number;
};

type ContentMode = 'leaf' | 'list' | 'shelf';

export const DynamicPage: FC<DynamicPageProps> = ({ focusIndex }) => {
  const [mode, setMode] = useState<ContentMode>('shelf');

  let content: JSX.Element;
  switch (mode) {
    case 'shelf':
      content = (
        <NodeNav focusIndex={focusIndex}>
          <SimpleShelf focusIndex={0} />
        </NodeNav>
      );
      break;
    case 'leaf':
      content = <Leaf focusIndex={focusIndex} />;
      break;
    case 'list':
      content = (
        <VerticalNav elementCount={5} focusIndex={focusIndex}>
          <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
            {Array.from({ length: 5 }).map((_, i) => (
              <Leaf focusIndex={i} key={i} />
            ))}
          </Layout>
        </VerticalNav>
      );
      break;
  }

  return (
    <Layout fullWidth>
      <SegmentedButton>
        <SegmentedButtonOption
          defaultChecked
          label="Shelf"
          name="content-mode"
          onChange={(e) => {
            if (e.currentTarget.checked) {
              setMode('shelf');
            }
          }}
        />
        <SegmentedButtonOption
          label="Leaf"
          name="content-mode"
          onChange={(e) => {
            if (e.currentTarget.checked) {
              setMode('leaf');
            }
          }}
        />
        <SegmentedButtonOption
          label="List"
          name="content-mode"
          onChange={(e) => {
            if (e.currentTarget.checked) {
              setMode('list');
            }
          }}
        />
      </SegmentedButton>
      <Layout padding={{ top: 5 }}>{content}</Layout>
    </Layout>
  );
};
