import type { FC } from 'react';
import { useEffect } from 'react';
import { NavigationRoot } from 'tachyon-tv-nav';
import { useUserIntent } from 'tachyon-user-intent';
import { KeyValue, legacyKeyboardEventHandler } from 'tachyon-utils';

/**
 * This value is set to be slow enough that you can still comprehend animations
 * while holding down a direction key.
 */
const KEY_NAV_THROTTLE_WAIT = 100;

const SCROLL_NAV_THROTTLE_WAIT = 1000;

export const NavigationRootAdapter: FC<{ pageType: string }> = ({
  children,
  pageType,
}) => {
  const { triggerIntent } = useUserIntent();

  useEffect(() => {
    const navButtonListener = legacyKeyboardEventHandler((event) => {
      const key = event.key as KeyValue;
      switch (key) {
        case KeyValue.Down:
          triggerIntent('down');
          break;
        case KeyValue.Up:
          triggerIntent('up');
          break;
        case KeyValue.Right:
          triggerIntent('right');
          break;
        case KeyValue.Left:
          triggerIntent('left');
          break;
        default:
          return;
      }

      // Manufacturers such as Vestel build a psuedo tab-index system into their browser that
      // will respond to d-pad navigation. Disable input events from bubbling to those systems
      // in order to retain sole control of DOM focus management.
      event.preventDefault();
    });
    document.addEventListener('keydown', navButtonListener);

    return () => {
      document.removeEventListener('keydown', navButtonListener);
    };
  });

  return (
    <NavigationRoot
      focusSeed={pageType}
      keyNavThrottleMs={KEY_NAV_THROTTLE_WAIT}
      scrollNavThrottleMs={SCROLL_NAV_THROTTLE_WAIT}
    >
      {children}
    </NavigationRoot>
  );
};
