import type { FC } from 'react';
import { useMemo, useRef } from 'react';
import styled from 'styled-components';
import { HorizontalNav, useAreaChildFocusIndex } from 'tachyon-tv-nav';
import { useUserIntent } from 'tachyon-user-intent';
import {
  AlignItems,
  Background,
  Button,
  ButtonSize,
  CoreText,
  Display,
  FontSize,
  Layout,
  Overflow,
  Position,
  SVGAsset,
} from 'twitch-core-ui';
import { FocusableCard } from '../FocusableCard';

type ShelfProps = {
  count: number;
  pageSize: number;
};

const ScShelfLayout = styled.div`
  display: flex;
  overflow-x: auto;
  scroll-snap-type: x mandatory;
  width: 100%;
`;

const ScCardLayout = styled.div`
  flex-shrink: 0;
  margin: 0 0.5rem;
  scroll-snap-align: start;
  width: 40rem;
`;

export const VirtualizedShelfBase: FC<ShelfProps> = ({ count, pageSize }) => {
  const { triggerIntent } = useUserIntent();
  const focusedChildIndex = useAreaChildFocusIndex();
  const renderCount = useRef(0);
  renderCount.current++;

  const cards = useMemo(() => {
    const allCards = Array.from({ length: count }).map((_, i) => i);

    // Render a full "page" off content off screen to the left
    const startIdx = Math.max(0, focusedChildIndex - pageSize * 2);

    // Render a full on screen "page" and off screen page to the right
    const endIdx = Math.min(allCards.length, focusedChildIndex + pageSize * 2);

    return allCards.slice(startIdx, endIdx).map((absoluteIdx) => (
      <ScCardLayout
        key={absoluteIdx}
        ref={(e: HTMLDivElement) => {
          if (e && absoluteIdx === focusedChildIndex) {
            e.scrollIntoView({ behavior: 'smooth' });
          }
        }}
      >
        <FocusableCard focusIndex={absoluteIdx} />
      </ScCardLayout>
    ));
  }, [count, pageSize, focusedChildIndex]);

  return (
    <Layout position={Position.Relative}>
      <Layout
        alignItems={AlignItems.Center}
        attachBottom
        attachLeft
        attachTop
        display={Display.Flex}
        margin={{ x: 2 }}
        position={Position.Absolute}
      >
        <Button
          icon={SVGAsset.AngleLeft}
          onClick={() => triggerIntent('page-left')}
          size={ButtonSize.Large}
        />
      </Layout>
      <Layout
        alignItems={AlignItems.Center}
        attachBottom
        attachRight
        attachTop
        display={Display.Flex}
        margin={{ x: 2 }}
        position={Position.Absolute}
      >
        <Button
          icon={SVGAsset.AngleRight}
          onClick={() => triggerIntent('page-right')}
          size={ButtonSize.Large}
        />
      </Layout>
      <Layout overflow={Overflow.Scroll}>
        <CoreText fontSize={FontSize.Size3}>
          {`Shelf: Renders ${renderCount.current}, Focus Idx: ${focusedChildIndex}`}
        </CoreText>
        <Layout
          background={Background.Accent}
          fullWidth
          margin={{ y: 1 }}
          overflow={Overflow.Hidden}
        >
          <ScShelfLayout>{cards}</ScShelfLayout>
        </Layout>
      </Layout>
    </Layout>
  );
};

export const VirtualizedShelf: FC<ShelfProps & { focusIndex: number }> = (
  props,
) => {
  return (
    <HorizontalNav
      elementCount={props.count}
      focusIndex={props.focusIndex}
      pageSize={props.pageSize}
    >
      <VirtualizedShelfBase {...props} />
    </HorizontalNav>
  );
};
