import type { FC } from 'react';
import { useRef, useState } from 'react';
import ReactDOM from 'react-dom';
import { CoreUiSsrRoot } from 'tachyon-more-ui';
import { NodeNav, VerticalNav, useFocus } from 'tachyon-tv-nav';
import { UserIntentRoot } from 'tachyon-user-intent';
import { useStorage } from 'tachyon-utils';
import type { TabItemPublicProps } from 'twitch-core-ui';
import {
  AlignItems,
  Background,
  Button,
  CheckBox,
  Color,
  CoreText,
  Display,
  FlexDirection,
  Layout,
  Overflow,
  TabSize,
  Tabs,
} from 'twitch-core-ui';
import { DynamicPage } from './DynamicPage';
import { GridPage } from './GridPage';
import { NavigationRootAdapter } from './NavigationRootAdapter';
import { ShelvesPage } from './ShelvesPage';
import { FocusableButton } from './common';

const NavBar: FC<{ focusIndex: number }> = ({ focusIndex }) => {
  const { focusId, focused } = useFocus(focusIndex);
  const renderCount = useRef(0);
  renderCount.current++;

  return (
    <Layout
      background={focused ? Background.AccentAlt2 : Background.Alt}
      color={focused ? Color.Overlay : Color.Base}
    >
      <CoreText>{`Cool Top Nav: Id: ${focusId} Renders ${renderCount.current}`}</CoreText>
    </Layout>
  );
};

type PageType = 'dynamic' | 'grid' | 'shelves';

export const TVNavExample: FC = () => {
  const [pageType, setPageType] = useStorage<PageType>('pageType', 'shelves');
  const [handleWheel, setHandleWheel] = useStorage('handleWheel', false);

  const [showError, setShowError] = useState(false);

  function toggleHandleWheel(): void {
    setHandleWheel(!handleWheel);
  }

  let page: { content: JSX.Element; index: number };
  switch (pageType) {
    case 'shelves':
      page = {
        content: <ShelvesPage focusIndex={1} handleWheel={handleWheel} />,
        index: 0,
      };
      break;
    case 'grid':
      page = {
        content: <GridPage focusIndex={1} handleWheel={handleWheel} />,
        index: 1,
      };
      break;
    case 'dynamic':
      page = { content: <DynamicPage focusIndex={1} />, index: 2 };
      break;
  }

  const exampleTabs: TabItemPublicProps[] = [
    {
      label: 'Shelves Example',
      onClick: () => {
        setPageType('shelves');
      },
    },
    {
      label: 'Grid Example',
      onClick: () => {
        setPageType('grid');
      },
    },
    {
      label: 'Dynamic Example',
      onClick: () => {
        setPageType('dynamic');
      },
    },
  ];

  return (
    <CoreUiSsrRoot appRootElementId="root">
      <UserIntentRoot>
        <NavigationRootAdapter pageType={pageType}>
          <Layout
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            fullWidth
            overflow={Overflow.Hidden}
          >
            <Layout
              alignItems={AlignItems.Center}
              display={Display.Flex}
              padding={{ x: 1, y: 1 }}
            >
              <Layout padding={{ x: 1 }}>
                <CheckBox
                  defaultChecked={handleWheel}
                  label="Scroll Wheel"
                  onChange={toggleHandleWheel}
                />
              </Layout>
              <Button
                onClick={() => {
                  setShowError(true);
                }}
              >
                Show Error
              </Button>
            </Layout>
            <Tabs
              activeTabIndex={page.index}
              size={TabSize.Large}
              tabs={exampleTabs}
            />
            <VerticalNav elementCount={2} focusIndex={0}>
              <Layout padding={{ top: 5 }}>
                <NavBar focusIndex={0} />
                {showError ? (
                  <NodeNav focusIndex={1}>
                    <FocusableButton
                      focusIndex={0}
                      onClick={() => {
                        setShowError(false);
                      }}
                    >
                      Dismiss Error
                    </FocusableButton>
                  </NodeNav>
                ) : (
                  <Layout padding={{ top: 5 }}>{page.content}</Layout>
                )}
              </Layout>
            </VerticalNav>
          </Layout>
        </NavigationRootAdapter>
      </UserIntentRoot>
    </CoreUiSsrRoot>
  );
};

ReactDOM.render(<TVNavExample />, document.getElementById('root'));
