import type { FormEvent, KeyboardEvent } from 'react';
import { Component } from 'react';
import styled, { createGlobalStyle } from 'styled-components';
import { uniqueIDGenerator } from 'tachyon-utils';
import type {
  TMIBadgeLoot,
  TMIBitsLoot,
  TMIChatActionEvent,
  TMIChatMessage,
  TMIChatMessageEvent,
  TMIChatNoticeEvent,
  TMICrateEvent,
  TMIEmoteLoot,
  TMIInGameContentLoot,
  TMILogger,
  TMIPurchaseEvent,
  TMIRewardGiftEvent,
  TMIRoomState,
} from 'twitch-chat-client';
import {
  TMIChatEventType,
  TMIClient,
  TMIConnectionState,
  TMICrateLootType,
} from 'twitch-chat-client';
import {
  Button,
  ButtonType,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  Overflow,
  Position,
  TextArea,
} from 'twitch-core-ui';

interface TMIPlaygroundState {
  chatBuffer: Array<
    | TMIChatMessageEvent
    | TMIChatNoticeEvent
    | TMICrateEvent
    | TMIPurchaseEvent
    | TMIRewardGiftEvent
  >;
  chatInputValue: string;
  roomState: TMIRoomState;
}

/* eslint-disable react/forbid-dom-props */

// TODO:
const ScGlobalStyle = createGlobalStyle`
  html {
    font-size: 14px;
    height: 100%;
  }

  body {
    height: 100%;
  }

  #root {
    height: 100%;
  }

  .chat-message {
    border-bottom: 1px solid #eee;
    padding-bottom: 8px;
    padding-top: 8px;
  }

  .chat-controls {
    background: #eee;
    border-top: 1px solid #ddd;
    display: flex;
    flex-direction: column;
    left: 0;
    padding: 10px;
    position: absolute;
    right: 0;
    top: 100%;
  }

  .chat-author {
    font-weight: bold;
  }

  .chat-purchase-boxart {
    max-height: 5.5rem;
    max-width: 4rem;
  }

  .chat-purchase-title {
    position: absolute;
  }

  .chat-purchase-reward {
    margin-left: 2px;
  }

  .chat-purchase-loot {
    max-height: 2.8rem;
    max-width: 2.8rem;
  }

  .chat-controls input {
    font-size: 16px;
    height: 40px;
  }

  .chat-message-debug {
    display: flex;
    font-size: 11px;
  }

  .chat-message-debug > div {
    border-right: 1px solid #eee;
    margin-right: 4px;
    padding-right: 4px;
  }
`;

const ScChatContainer = styled(Layout)`
  height: 700px;
  position: relative;
  width: 700px;
`;

export type ChatConsumerProps = {
  channel: string;
  currentUser: { authToken: string; login: string } | null;
};

export class ChatConsumer extends Component<
  ChatConsumerProps,
  TMIPlaygroundState
> {
  private channel = '';
  private currentUser: ChatConsumerProps['currentUser'] = null;
  private client: TMIClient | null = null;
  private reconnectHandle = 0;
  private reconnectTestEnabled = false;
  private isReconnecting = false;

  constructor(props: ChatConsumerProps) {
    super(props);

    this.channel = props.channel;
    this.currentUser = props.currentUser;

    this.state = {
      chatBuffer: [],
      chatInputValue: '',
      roomState: {
        broadcasterLang: null,
        emoteOnly: false,
        followersOnly: false,
        mercury: false,
        r9k: false,
        slowMode: false,
        subsOnly: false,
      },
    };

    this.establishConnection();
  }

  private async establishConnection(): Promise<void> {
    const logger: TMILogger = {
      debug(message: string, args: unknown[]) {
        console.debug(message, args);
      },
      error(error: Error, message: string, ...args: unknown[]) {
        console.warn(message, error, args);
      },
      info(message: string, args: unknown[]) {
        // Nothing
        console.info(message, args);
      },
      trackAndWarn(
        _: string,
        __: Record<string, unknown>,
        message: string,
        ...args: unknown[]
      ) {
        this.warn(message, args);
      },
      warn(message: string, args: unknown[]) {
        console.warn(message, args);
      },
    };

    this.reconnectHandle = 0;
    const client = new TMIClient({
      connection: {
        port: 443,
        secure: true,
        server: 'irc-ws.chat.twitch.tv',
      },
      identity: {
        authToken: this.currentUser?.authToken ?? '',
        username: this.currentUser?.login ?? '',
      },
      logger,
    });

    client.events.connecting((d) => {
      console.log(`Connecting to ${d.address}:${d.port}`);
    });

    client.events.reconnecting(() => {
      this.addSystemMessage('Reconnecting you...');
    });

    client.events.reconnected(() => {
      this.addSystemMessage('Reconnected!');
    });

    client.events.followersonly((d) => {
      if (d.enabled) {
        this.addSystemMessage(
          `This room is now in followers only mode for ${d.length} mins`,
        );
      } else {
        this.addSystemMessage(`This room is no longer in follower only mode`);
      }
    });

    client.events.subscribers((d) => {
      if (d.enabled) {
        this.addSystemMessage(`This room is now in subs only mode`);
      } else {
        this.addSystemMessage(`This room is no longer in subs only mode`);
      }
    });

    client.events.slowmode((d) => {
      if (d.enabled) {
        this.addSystemMessage(
          `This room is now in slow mode for ${d.length} mins`,
        );
      } else {
        this.addSystemMessage(`This room is no longer in slow mode`);
      }
    });

    client.events.connected((d) => {
      const duration = d.timestampJoined - d.timestampCreated;
      this.onTriggerJoin();
      this.addSystemMessage(
        `Welcome to the chat! It took us ${duration} ms to get you connected.`,
      );
      console.log('Connected via event!');
    });

    client.events.joined((data) => {
      this.addSystemMessage(`You have joined ${data.channel}`);
      // Try to turn on follower only mode
      if (this.reconnectTestEnabled && !this.reconnectHandle) {
        console.info('Starting reconnect interval!');
        this.reconnectHandle = window.setInterval(this.reconnectTest, 5000);
      }
    });

    client.events.parted((data) => {
      this.addSystemMessage(`You have parted ${data.channel}`);
    });

    client.events.disconnected((data) => {
      this.addSystemMessage(`You have disconnected: ${data.reason}`);
    });

    client.events.roomstate((data) => {
      console.info('roomstate', data);
      this.setState({
        roomState: data.state,
      });
    });

    client.events.chat((data) => {
      this.addChatMessage(data);
    });

    client.events.usernotice((data) => {
      console.info('usernotice', data);
      this.addChatMessage(data);
    });

    client.events.action((data) => {
      this.addChatMessage(data);
    });

    client.events.notice((data) => {
      console.debug('Chat notice', data);
      this.addChatMessage(data);
    });

    client.events.subscription((data) => {
      console.debug('Subscription Event', data);
      const str = `${data.user.displayName} Subscribed at Tier ${data.methods.plan}`;
      this.addSystemMessage(str);
    });

    client.events.resub((data) => {
      console.debug('Resubscription Gift Event', data);
      let str = `${data.user.displayName} Subscribed at Tier ${data.methods.plan} for ${data.cumulativeMonths} months`;
      if (data.shouldShareStreakTenure) {
        str = `${data.user.displayName} Subscribed at Tier ${data.methods.plan} for ${data.cumulativeMonths} months, currently on a ${data.streakMonths} month streak!`;
      }
      this.addSystemMessage(str);
    });

    client.events.extendsub((data) => {
      console.debug('ExtendSubscription Event', data);
      const str = `${data.user.displayName} extended their Tier ${data.tier} subscription through ${data.benefitEndMonth}!`;
      this.addSystemMessage(str);
    });

    client.events.subgift((data) => {
      console.debug('Subscription Gift Event', data);
      let str = `${data.user.displayName} gifted a Tier ${data.methods.plan} Sub to ${data.recipientName}!`;
      if (data.senderCount !== 0) {
        str = `${str} They've given ${data.senderCount} Gift Subs in the channel!`;
      }
      if (data.giftMonths > 1) {
        str = `${data.user.displayName} gifted ${data.giftMonths} months of Tier ${data.methods.plan} to ${data.recipientName}!`;
        if (data.senderCount !== 0) {
          str = `${str} They've gifted ${data.senderCount} months in the channel!`;
        }
      }
      this.addSystemMessage(str);
    });

    client.events.anonsubgift((data) => {
      console.debug('Anon Subscription Gift Event', data);
      let str = `An anonymous gifter gave ${data.recipientName} a Tier ${data.methods.plan} Sub!`;
      if (data.giftMonths > 1) {
        str = `An anonymous gifter gifted ${data.giftMonths} months of Tier ${data.methods.plan} to ${data.recipientName}!`;
      }
      if (data.funString) {
        str = `${str} ${data.funString}`;
      }
      this.addSystemMessage(str);
    });

    client.events.submysterygift((data) => {
      console.debug('Subscription Mystery Gift Event', data);
      let str = `${data.user.displayName} is gifting ${data.massGiftCount} Tier ${data.plan} Subs to ${data.channel}'s community!`;
      if (data.senderCount !== 0) {
        str = `${str} They've gifted a total of ${data.senderCount} in the channel!`;
      }
      this.addSystemMessage(str);
    });

    client.events.anonsubmysterygift((data) => {
      console.debug('Anon Subscription Mystery Gift Event', data);
      let str = `An anonymous gifter is gifting ${data.massGiftCount} Tier ${data.plan} Subs to ${data.channel}'s community!`;
      if (data.funString) {
        str = `${str} ${data.funString}`;
      }
      this.addSystemMessage(str);
    });

    client.events.giftpaidupgrade((data) => {
      console.debug('Gift Paid Upgrade Event', data);
      let str = `${data.user.username} is continuing the Gift Sub they got from ${data.senderName}!`;
      if (data.promoGiftTotal && data.promoName) {
        str = `${str} They're one of ${data.promoGiftTotal} gift subs to continue this ${data.promoName}`;
      }
      this.addSystemMessage(str);
    });

    client.events.anongiftpaidupgrade((data) => {
      console.debug('Anon Gift Paid Upgrade Event', data);
      let str = `${data.user.username} is continuing the Gift Sub they got from an anonymous gifter!`;
      if (data.promoGiftTotal && data.promoName) {
        str = `${str} They're one of ${data.promoGiftTotal} gift subs to continue this ${data.promoName}`;
      }
      this.addSystemMessage(str);
    });

    client.events.communitypayforward((data) => {
      console.debug('Community Sub Gift Pay Forward Event', data);
      this.addSystemMessage(
        `${data.user.username} is paying forward the Gift they got from ${data.priorGifterName} to the community!`,
      );
    });

    client.events.standardpayforward((data) => {
      console.debug('Standard Sub Gift Pay Forward Event', data);
      this.addSystemMessage(
        `${data.user.username} is paying forward the Gift they got from ${data.priorGifterName} to ${data.recipientName}!`,
      );
    });

    client.events.primepaidupgrade((data) => {
      console.debug('Prime Paid Upgrade Event', data);
      this.addSystemMessage(
        `${data.user.username} converted from a Twitch Prime sub to a Tier ${data.plan} sub!`,
      );
    });

    client.events.charity((data) => {
      console.debug('Bits Charity Event', data);
      this.addSystemMessage(
        `Bits Total: ${data.total} Hours Left: ${data.hoursLeft} Days Left: ${data.daysLeft} Charity Name: ${data.charityName}`,
      );
    });

    client.events.purchase((data) => {
      this.addChatMessage(data);
    });

    client.events.crate((data) => {
      this.addChatMessage(data);
    });

    client.events.rewardgift((data) => {
      this.addChatMessage(data);
    });

    client.events.mods((data) => {
      this.addSystemMessage(
        `The mods for this channel are: ${data.usernames.join(', ')}`,
      );
    });

    client.events.badgesupdated((data) => {
      const badgeData = Object.keys(data.badges).join(', ');
      this.addSystemMessage(`Badges have been updated: ${badgeData}`);
    });

    client.events.emoteonlymode((data) => {
      this.addSystemMessage(
        data.enabled
          ? 'This room is now in emote only mode.'
          : 'This room is no longer in emote only mode.',
      );
    });

    client.events.celebrationpurchase((data) => {
      console.debug('Celebration Purchase Event', data);
      this.addSystemMessage(
        `${data.user.username} is about to launch ${data.intensity} ${data.effect}`,
      );
    });

    const result = await client.connect();
    if (result.state !== TMIConnectionState.Connected) {
      console.log('Failed to connect!', result);
    }

    this.client = client;
  }

  public override render(): JSX.Element {
    const roomState = this.state.roomState;

    return (
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Row}
        fullHeight
      >
        <ScChatContainer
          flexDirection={FlexDirection.Column}
          justifyContent={JustifyContent.Between}
        >
          <Layout
            attachBottom
            attachTop
            fullWidth
            overflow={Overflow.Auto}
            padding={1}
            position={Position.Absolute}
          >
            {this.state.chatBuffer.map((m) => this.renderChatEvent(m))}
          </Layout>
          <div className="chat-controls">
            <TextArea
              onChange={this.onInputChanged}
              onKeyDown={this.onKeyDown}
              value={this.state.chatInputValue}
            />
          </div>
        </ScChatContainer>
        <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
          <Layout padding={0.5}>
            <Button
              onClick={this.toggleFollowersOnly}
              variant={ButtonType.Secondary}
            >
              {roomState.followersOnly
                ? 'Disable Followers Only Mode'
                : 'Enable Followers Only Mode'}
            </Button>
          </Layout>
          <Layout padding={0.5}>
            <Button
              onClick={this.toggleSubsOnly}
              variant={ButtonType.Secondary}
            >
              {roomState.subsOnly
                ? 'Disable Subs Only Mode'
                : 'Enable Subs Only Mode'}
            </Button>
          </Layout>
          <Layout padding={0.5}>
            <Button
              onClick={this.toggleSlowMode}
              variant={ButtonType.Secondary}
            >
              {roomState.slowMode ? 'Disable Slow Mode' : 'Enable Slow Mode'}
            </Button>
          </Layout>
          <Layout padding={0.5}>
            <Button
              onClick={this.onTriggerArtificialDisconnect}
              variant={ButtonType.Secondary}
            >
              Simulate Disconnect
            </Button>
          </Layout>
          <Layout padding={0.5}>
            <Button
              onClick={this.onTriggerDisconnect}
              variant={ButtonType.Secondary}
            >
              Disconnect
            </Button>
          </Layout>
          <Layout padding={0.5}>
            <Button
              onClick={this.onTriggerConnect}
              variant={ButtonType.Secondary}
            >
              Connect
            </Button>
          </Layout>
          <Layout padding={0.5}>
            <Button
              onClick={this.onTriggerReconnect}
              variant={ButtonType.Secondary}
            >
              Reconnect
            </Button>
          </Layout>
          <Layout padding={0.5}>
            <Button onClick={this.onTriggerPart} variant={ButtonType.Secondary}>
              Part
            </Button>
          </Layout>
          <Layout padding={0.5}>
            <Button onClick={this.onClickInject} variant={ButtonType.Secondary}>
              Inject
            </Button>
          </Layout>
          <Layout padding={0.5}>
            <Button onClick={this.onTriggerJoin} variant={ButtonType.Primary}>
              Join
            </Button>
          </Layout>
        </Layout>
        <ScGlobalStyle />
      </Layout>
    );
  }

  private onInputChanged = (evt: FormEvent<HTMLTextAreaElement>) => {
    this.setState({
      chatInputValue: (evt.target as HTMLTextAreaElement).value,
    });
  };

  private onKeyDown = (event: KeyboardEvent<HTMLTextAreaElement>) => {
    const key = event.charCode || event.keyCode;
    if (key === 13) {
      event.preventDefault();
      this.sendChatMessage();
    }
  };

  private sendChatMessage = async () => {
    const message = this.state.chatInputValue;
    this.setState({ chatInputValue: '' });
    try {
      const result = await this.client?.sendCommand(this.channel, message, {});
      console.log('sendChatMessage', { result });
    } catch (err) {
      console.error('Failed to get response within timeout');
    }
  };

  private onTriggerArtificialDisconnect = () => {
    this.client?.triggerArtificialDisconnect();
  };

  private onTriggerDisconnect = () => {
    this.client?.disconnect();
  };

  private onTriggerConnect = () => {
    this.client?.connect();
  };

  private onTriggerReconnect = () => {
    this.client?.reconnect();
  };

  private onTriggerPart = async () => {
    try {
      await this.client?.partChannel(this.channel);
    } catch (err) {
      this.addSystemMessage('Failed to part channel');
    }
  };

  private onTriggerJoin = async () => {
    try {
      await this.client?.joinChannel(this.channel);
    } catch (err) {
      this.addSystemMessage('Failed to join channel');
    }
  };

  private onClickInject = () => {
    const message = this.state.chatInputValue;
    this.setState({ chatInputValue: '' });
    this.client?.injectMessage(message);
  };

  private toggleFollowersOnly = async () => {
    if (this.state.roomState.followersOnly) {
      await this.client?.commands.followersOnlyOff.execute({
        channel: this.channel,
      });
    } else {
      await this.client?.commands.followersOnlyOn.execute({
        channel: this.channel,
      });
    }
  };

  private toggleSubsOnly = async () => {
    if (this.state.roomState.subsOnly) {
      await this.client?.commands.subscriberModeOff.execute({
        channel: this.channel,
      });
    } else {
      await this.client?.commands.subscriberModeOn.execute({
        channel: this.channel,
      });
    }
  };

  private toggleSlowMode = async () => {
    if (this.state.roomState.slowMode) {
      await this.client?.commands.slowModeOff.execute({
        channel: this.channel,
      });
    } else {
      await this.client?.commands.slowModeOn.execute({ channel: this.channel });
    }
  };

  private renderChatEvent(
    chatEvent:
      | TMIChatActionEvent
      | TMIChatMessageEvent
      | TMIChatNoticeEvent
      | TMICrateEvent
      | TMIPurchaseEvent
      | TMIRewardGiftEvent,
  ) {
    switch (chatEvent.type) {
      case TMIChatEventType.Message:
      case TMIChatEventType.UserNotice:
        return this.renderChatMessage(chatEvent as TMIChatMessageEvent);
      case TMIChatEventType.Notice:
        return this.renderChatNotice(chatEvent as TMIChatNoticeEvent);
      case TMIChatEventType.Action:
        return this.renderChatMessage(chatEvent as TMIChatActionEvent);
      case TMIChatEventType.Purchase:
        return this.renderChatMessage(chatEvent as TMIPurchaseEvent);
      case TMIChatEventType.Crate:
        return this.renderChatMessage(chatEvent as TMICrateEvent);
      case TMIChatEventType.RewardGift:
        return this.renderChatMessage(chatEvent as TMIRewardGiftEvent);
      default:
        console.warn('Unknown chat event type', { type: chatEvent.type });
        break;
    }
  }

  private renderChatNotice(notice: TMIChatNoticeEvent) {
    return (
      <div key={notice.timestamp}>
        <span>{notice.body}</span>
      </div>
    );
  }

  private renderChatMessage(
    chatEvent:
      | TMIChatActionEvent
      | TMIChatMessageEvent
      | TMICrateEvent
      | TMIPurchaseEvent
      | TMIRewardGiftEvent,
  ) {
    const message = chatEvent.message;
    const emotes = JSON.stringify(message.user.emotes) || 'None';
    const badges = JSON.stringify(message.user.badges) || 'None';

    return (
      <div className="chat-message" key={message.user.id}>
        {this.renderMessageBody(chatEvent)}
        <div className="chat-message-debug">
          <div>
            <b>Timestamps:</b> {chatEvent.timestamp} (Client){' '}
            {message.timestamp} (Server)
          </div>
          <div>
            <b>Color:</b> {message.user.color || 'None'}
          </div>
          <div>
            <b>User:</b> {message.user.userID || 'No User ID'},{' '}
            {message.user.username || 'No Username'}, {message.user.displayName}{' '}
            ({message.user.userType})
          </div>
          <div>
            <b>Turbo:</b> {JSON.stringify(message.user.turbo)}
          </div>
          <div>
            <b>Bits:</b> {message.user.bits}
          </div>
          <div>
            <b>Emotes:</b> {emotes}
          </div>
          <div>
            <b>Badges:</b> {badges}
          </div>
        </div>
      </div>
    );
  }

  private renderMessageBody(
    chatEvent:
      | TMIChatActionEvent
      | TMIChatMessageEvent
      | TMICrateEvent
      | TMIPurchaseEvent
      | TMIRewardGiftEvent,
  ) {
    if (chatEvent.type === TMIChatEventType.Action) {
      return this.renderChatActionBody(chatEvent as TMIChatActionEvent);
    } else if (chatEvent.type === TMIChatEventType.Purchase) {
      return this.renderChatPurchaseBody(chatEvent as TMIPurchaseEvent);
    } else if (chatEvent.type === TMIChatEventType.Crate) {
      return this.renderChatCrateBody(chatEvent as TMICrateEvent);
    }
    return this.renderChatMessageBody(chatEvent.message);
  }

  private renderChatActionBody(chatEvent: TMIChatActionEvent) {
    const { message } = chatEvent;
    return (
      <div>
        <span className="chat-author" style={{ color: message.user.color }}>
          {message.user.displayName} {chatEvent.action}
        </span>
      </div>
    );
  }

  private renderChatMessageBody(message: TMIChatMessage) {
    return (
      <div>
        <span className="chat-author" style={{ color: message.user.color }}>
          {message.user.displayName}
        </span>
        : <span>{message.body}</span>
      </div>
    );
  }

  private renderChatCrateBody(crateEvent: TMICrateEvent) {
    return (
      <span>
        <strong>{crateEvent.message.user.displayName}</strong> triggered{' '}
        {crateEvent.selectedCount} crate gifts to the channel
      </span>
    );
  }

  private renderChatPurchaseBody(purchaseEvent: TMIPurchaseEvent) {
    const { message, purchase } = purchaseEvent;
    const { user } = message;
    const { crateLoot, numCrates, purchased } = purchase;

    const rewardText =
      numCrates === 0 ? (
        ''
      ) : (
        <span className="chat-purchase-reward">
          Plus {numCrates} Twitch Crates with {crateLoot.length} rewards!
        </span>
      );

    const lootList = crateLoot.map((lootItem) => {
      switch (lootItem.type) {
        case TMICrateLootType.Badge:
          return this.renderBadgeLoot(lootItem);
        case TMICrateLootType.Bits:
          return this.renderBitsLoot(lootItem);
        case TMICrateLootType.Emote:
          return this.renderEmoteLoot(lootItem);
        case TMICrateLootType.InGameContent:
          return this.renderIGCLoot(lootItem);
        default:
          return '';
      }
    });

    const loot = crateLoot.length === 0 ? '' : <span>{lootList}</span>;

    return (
      <div>
        <div>
          {user.username} bought {purchased.title}!
        </div>
        <span className="chat-author" style={{ color: user.color }}>
          {user.displayName}
        </span>
        : <span>{message.body}</span>
        <div className="chat-purchase">
          <span>
            <img className="chat-purchase-boxart" src={purchased.boxart} />
            <span className="chat-purchase-title">{purchased.title}</span>
            {rewardText}
          </span>
          {loot}
        </div>
      </div>
    );
  }

  private renderBadgeLoot(loot: TMIBadgeLoot) {
    return <img className="chat-purchase-loot" src={loot.img} />;
  }

  private renderBitsLoot(loot: TMIBitsLoot) {
    return <span>{loot.quantity} bits</span>;
  }

  private renderEmoteLoot(loot: TMIEmoteLoot) {
    const src = `//static-cdn.jtvnw.net/emoticons/v1/${loot.id}/2.0`;
    return <img className="chat-purchase-loot" src={src} />;
  }

  private renderIGCLoot(loot: TMIInGameContentLoot) {
    return <img className="chat-purchase-loot" src={loot.img} />;
  }

  private reconnectTest = async () => {
    try {
      if (this.isReconnecting) {
        console.warn('Already reconnecting!');
        return;
      }

      this.isReconnecting = true;
      this.addSystemMessage('Reconnecting you...');
      try {
        await this.client?.reconnect();
      } catch (err) {
        console.warn('Reconnect failed!', err);
      }
      this.addSystemMessage('You are reconnected!');
      // const onRes = await client.commands.followersOnly.execute({ channel });
      // const offRes = await client.commands.followersOnlyOff.execute({ channel });
    } catch (err) {
      console.error('Failed to execute followers only chat', err);
    }
    this.isReconnecting = false;
  };

  private addSystemMessage = (message: string) => {
    const systemMessage: TMIChatMessageEvent = {
      channel: this.channel,
      message: {
        body: message,
        id: uniqueIDGenerator(),
        timestamp: Date.now(),
        user: {
          badgeDynamicData: null,
          badges: null,
          bits: 0,
          color: '',
          displayName: 'System',
          emotes: {},
          id: uniqueIDGenerator(),
          turbo: false,
          userID: '0',
          userType: 'system',
          username: 'system',
        },
      },
      sentByCurrentUser: false,
      timestamp: Date.now(),
      type: TMIChatEventType.Message,
    };

    this.addChatMessage(systemMessage);
  };

  private addChatMessage(
    message:
      | TMIChatActionEvent
      | TMIChatMessageEvent
      | TMIChatNoticeEvent
      | TMICrateEvent
      | TMIPurchaseEvent
      | TMIRewardGiftEvent,
  ) {
    this.setState((prevState) => {
      const messages = [...prevState.chatBuffer];
      if (messages.length > 20) {
        messages.shift();
      }
      messages.push(message);
      return { chatBuffer: messages };
    });
  }
}
