import type { FC } from 'react';
import { useCallback, useEffect, useRef, useState } from 'react';
import ReactDOM from 'react-dom';
import { CoreUiSsrRoot } from 'tachyon-more-ui';
import { useStorage } from 'tachyon-utils';
import {
  AlignItems,
  Button,
  CheckBox,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
} from 'twitch-core-ui';

// uses media like Twitch-created content that won't be deleted
const MEDIA = {
  clip: {
    id: 'AnimatedSilkyBoarRaccAttack',
    name: 'DJ Wheat',
  },
  stream: {
    id: 'monstercat',
    name: 'Monstercat',
  },
  vod: {
    id: '488672797',
    name: 'TwitchCon',
  },
} as const;

interface ControlButton {
  onClick: () => void;
  title: string;
}

type ButtonGenerator = (
  sendMessage: (command: string, data?: any) => void,
) => ControlButton[];

const getPlaybackButtons: ButtonGenerator = (sendMessage) => [
  { onClick: () => sendMessage('play'), title: 'Play' },
  { onClick: () => sendMessage('pause'), title: 'Pause' },
  { onClick: () => sendMessage('stop'), title: 'Stop' },
];

const getMuteButtons: ButtonGenerator = (sendMessage) => [
  { onClick: () => sendMessage('getMuted'), title: 'Get Muted' },
  { onClick: () => sendMessage('setMuted', true), title: 'Mute' },
  { onClick: () => sendMessage('setMuted', false), title: 'Unmute' },
];

const getVolumeButtons: ButtonGenerator = (sendMessage) => [
  { onClick: () => sendMessage('getVolume'), title: 'Get Volume' },
  {
    onClick: () => sendMessage('setVolume', 0.1),
    title: 'Volume to 0.1',
  },
  { onClick: () => sendMessage('setVolume', 1), title: 'Volume to 1' },
];

const getPositionButtons: ButtonGenerator = (sendMessage) => [
  { onClick: () => sendMessage('getPosition'), title: 'Get Position' },
  { onClick: () => sendMessage('setPosition', 10), title: 'Position to 10' },
  {
    onClick: () => sendMessage('setPosition', 1),
    title: 'Position to 1',
  },
];

const getStreamButtons: ButtonGenerator = (sendMessage) => [
  {
    onClick: () => sendMessage('load', { id: MEDIA.stream.id, type: 'stream' }),
    title: `Stream: ${MEDIA.stream.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.stream.id,
        options: { autoPlay: false },
        type: 'stream',
      }),
    title: `Stream (No autoPlay): ${MEDIA.stream.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.stream.id,
        options: { startMuted: false },
        type: 'stream',
      }),
    title: `Stream (Unmuted): ${MEDIA.stream.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.stream.id,
        options: { initialVolume: 0.5 },
        type: 'stream',
      }),
    title: `Stream (Volume: 0.5): ${MEDIA.stream.name}`,
  },
];

const getClipButtons: ButtonGenerator = (sendMessage) => [
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.clip.id,
        type: 'clip',
      }),
    title: `Clip: ${MEDIA.clip.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.clip.id,
        options: { autoPlay: false },
        type: 'clip',
      }),
    title: `Clip (No autoPlay): ${MEDIA.clip.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.clip.id,
        options: { loop: false },
        type: 'clip',
      }),
    title: `Clip (No loop): ${MEDIA.clip.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.clip.id,
        options: { startMuted: false },
        type: 'clip',
      }),
    title: `Clip (Unmuted): ${MEDIA.clip.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.clip.id,
        options: { initialVolume: 0.5 },
        type: 'clip',
      }),
    title: `Clip (Volume: 0.5): ${MEDIA.clip.name}`,
  },
];

const getVodButtons: ButtonGenerator = (sendMessage) => [
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.vod.id,
        type: 'vod',
      }),
    title: `VOD: ${MEDIA.vod.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.vod.id,
        options: { autoPlay: false },
        type: 'vod',
      }),
    title: `VOD (No autoPlay): ${MEDIA.vod.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.vod.id,
        options: { startFromTime: 300 },
        type: 'vod',
      }),
    title: `VOD (Start @ 5min): ${MEDIA.vod.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.vod.id,
        options: { startMuted: false },
        type: 'vod',
      }),
    title: `VOD (unmuted): ${MEDIA.vod.name}`,
  },
  {
    onClick: () =>
      sendMessage('load', {
        id: MEDIA.vod.id,
        options: { initialVolume: 0.5 },
        type: 'vod',
      }),
    title: `VOD (Volume: 0.5): ${MEDIA.vod.name}`,
  },
];

const orderedButtonGroups = [
  getPlaybackButtons,
  getMuteButtons,
  getVolumeButtons,
  getPositionButtons,
  getStreamButtons,
  getClipButtons,
  getVodButtons,
];

const targetUrl =
  process.env.REACT_APP_TARGET_URL || 'https://localhost.valence.tv.twitch.tv';
const ORIGIN = 'https://framehost.valence.tv.twitch.tv/';

const Example: FC = () => {
  const ref = useRef<HTMLIFrameElement | null>(null);
  const [messages, setMessages] = useState<string[]>([]);
  const [autoLoad, setAutoLoad] = useStorage<boolean>(
    'valence-auto-load',
    false,
  );

  const receiveMessage = useCallback(
    (event: MessageEvent) => {
      if (event.data && typeof event.data === 'string') {
        setMessages([...messages, event.data]);
      }
    },
    [messages],
  );

  useEffect(() => {
    // redirect to proper allowed host
    if (window.location.href !== ORIGIN) {
      window.location.href = ORIGIN;
    }
    window.addEventListener('message', receiveMessage);
    return () => window.removeEventListener('message', receiveMessage);
  }, [receiveMessage]);

  function sendMessage(command: string, data?: any): void {
    if (ref.current?.contentWindow) {
      const payload = data ? { command, data } : { command };
      ref.current.contentWindow.postMessage(JSON.stringify(payload), targetUrl);
    }
  }

  useEffect(() => {
    function autoMonstercat(event: MessageEvent): void {
      if (autoLoad && event.data && typeof event.data === 'string') {
        const data = JSON.parse(event.data);
        if (data.event === 'init') {
          console.log('auto-loading');
          // unmuted autoplay
          getStreamButtons(sendMessage)[1].onClick();
        }
      }
    }

    window.addEventListener('message', autoMonstercat);
    return () => window.removeEventListener('message', autoMonstercat);
  });

  function createButtonGroup(
    buttonGenerator: ButtonGenerator,
    index: number,
  ): JSX.Element {
    return (
      <Layout
        borderBottom
        display={Display.Flex}
        flexDirection={FlexDirection.Row}
        fullWidth
        justifyContent={JustifyContent.Between}
        key={index}
        padding={1}
      >
        {buttonGenerator(sendMessage).map(({ onClick, title }) => (
          <Button children={title} key={title} onClick={onClick} />
        ))}
      </Layout>
    );
  }

  return (
    <CoreUiSsrRoot appRootElementId="root">
      <main>
        <Layout display={Display.Flex}>
          <div>
            <Layout
              alignItems={AlignItems.Center}
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
            >
              <Layout
                display={Display.Flex}
                fullWidth
                justifyContent={JustifyContent.Between}
                padding={2}
              >
                <CoreText>{`iframe src: ${targetUrl}`}</CoreText>
                <CheckBox
                  checked={autoLoad}
                  label="Autoload Monstercat?"
                  onChange={(e) => {
                    setAutoLoad(e.currentTarget.checked);
                    window.location.reload();
                  }}
                />
              </Layout>
              <iframe
                allow="autoplay"
                height="360"
                ref={ref}
                src={targetUrl}
                title="valence"
                width="640"
              />
            </Layout>
            {orderedButtonGroups.map(createButtonGroup)}
          </div>
          <Layout borderLeft padding={1}>
            <Layout borderBottom fullWidth>
              <CoreText>Events</CoreText>
            </Layout>
            {messages.map((message, idx) => (
              <CoreText key={idx}>{message}</CoreText>
            ))}
          </Layout>
        </Layout>
      </main>
    </CoreUiSsrRoot>
  );
};

ReactDOM.render(<Example />, document.getElementById('root'));
