import type { FC, HTMLAttributes } from 'react';
import { useCallback, useEffect, useRef } from 'react';

interface ClickOutDetectorProps {
  /**
   * Called when a click event originates outside the container
   */
  onClickOut: (e: MouseEvent) => void;
}

export type Props = ClickOutDetectorProps & HTMLAttributes<HTMLDivElement>;

/**
 * This component creates a container element and notifies the owner when a
 * user clicks outside of it. This is useful for things like closing menus when
 * a user clicks anywhere else on the page.
 *
 * Usage:
 *
 *   <ClickOutDetector onClickOut={handleClickOut}>
 *    <h1>My Great Menu</h1>
 *   </ClickOutDetector>
 */
export const ClickOutDetector: FC<Props> = ({
  children,
  onClickOut,
  ...containerProps
}) => {
  // We don't destructure the props so we can get the containerProps later
  const container = useRef<HTMLDivElement | null>(null);

  const isParentOf = useCallback((target: HTMLElement): boolean => {
    let currentTarget: HTMLElement | null = target;

    while (currentTarget) {
      if (currentTarget === container.current) {
        return true;
      }

      currentTarget = currentTarget.parentElement;
    }

    return false;
  }, []);

  const handleGlobalMousedown = useCallback(
    (e: MouseEvent): void => {
      const target = e.target as HTMLElement;

      if (!isParentOf(target)) {
        onClickOut(e);
      }
    },
    [onClickOut, isParentOf],
  );

  useEffect(() => {
    // Capture this event before any children have a chance to.
    // This prevents child button actions from changing the DOM,
    // which could lead to a null target when this event is handled.
    document.addEventListener('mousedown', handleGlobalMousedown, true);

    return () => {
      document.removeEventListener('mousedown', handleGlobalMousedown, true);
    };
  });

  return (
    <div {...containerProps} ref={container}>
      {children}
    </div>
  );
};
ClickOutDetector.displayName = 'ClickOutDetector';
