import { startOfDay } from 'date-fns';
import Pikaday from 'pikaday';
import type { PikadayOptions } from 'pikaday';
import { useEffect } from 'react';
import type { FC } from 'react';
import { SCPikaday, SCtheme } from './date-picker';
import {
  useMonthNames,
  useNextMonthMessage,
  usePreviousMonthMessage,
  useWeekdayNames,
} from './utils';

interface DatePickerLiteProps {
  date: Date;
  endRange: Date;
  maxDate: Date;
  minDate?: Date;
  onChange: (date: Date) => void;
  startRange: Date;
}

/**
 * This DatePicker is a thin wrapper around the Pikaday library.
 *
 * https://github.com/dbushell/Pikaday
 *
 * This component should only provide a sensible component interface to Pikaday
 * and should not perform any validation or contain application logic. There
 * are no automated tests for this component since we do not own the interface
 * to Pikaday and cannot be confident that this component is working without
 * testing its integration against the full library in a real web browser.
 */
export const DatePickerLite: FC<DatePickerLiteProps> = (
  props: DatePickerLiteProps,
) => {
  let picker: Pikaday | null = null;
  const i18n = {
    months: useMonthNames(),
    nextMonth: useNextMonthMessage(),
    previousMonth: usePreviousMonthMessage(),
    weekdays: useWeekdayNames('long'),
    weekdaysShort: useWeekdayNames('short'),
  };

  useEffect(() => {
    if (!picker) {
      return;
    }
    // It's convenient for us to accept dates that are the end of a day, since
    // this fits with the model of how data is queried. For instance, selecting
    // Feb 1 in each picker should query for one whole day.
    //
    // However, Pikaday is picky about the extact time needed to set the date of
    // the picker: each date must be the beginning of the day.
    picker.setStartRange(startOfDay(props.startRange));
    picker.setEndRange(startOfDay(props.endRange));
    if (props.minDate) {
      picker.setMinDate(startOfDay(props.minDate));
    }
    picker.setMaxDate(startOfDay(props.maxDate));
    picker.setDate(startOfDay(props.date), true /* suppress onSelect */);
  }, [
    props.startRange,
    props.endRange,
    props.minDate,
    props.maxDate,
    props.date,
    picker,
  ]);

  const setContainerRef = (container: HTMLDivElement | null) => {
    if (container === null) {
      picker?.destroy();
    } else {
      picker = new Pikaday({
        defaultDate: startOfDay(props.date),
        endRange: startOfDay(props.endRange),
        i18n,
        maxDate: startOfDay(props.maxDate),
        minDate: props.minDate && startOfDay(props.minDate),
        onSelect: props.onChange,
        setDefaultDate: true,
        startRange: startOfDay(props.startRange),
        theme: 'inline',
        // Pikaday uses default on null for firstDay but the type doesn't match(type is undefined));
      } as PikadayOptions);

      container.appendChild(picker.el);
    }
  };
  return (
    <>
      <div ref={setContainerRef} />
      <SCPikaday />
      <SCtheme />
    </>
  );
};

DatePickerLite.displayName = 'DatePickerLite';
