import moment from 'moment';
import Pikaday from 'pikaday';
import type { FC } from 'react';
import { useEffect, useRef, useState } from 'react';
import { useIntl } from 'tachyon-intl';
import { useEffectOnce, usePrevious } from 'tachyon-utils-react';
import type { InputProps } from 'twitch-core-ui';
import {
  Display,
  InjectLayout,
  Input,
  InputSize,
  InputType,
  Layout,
} from 'twitch-core-ui';
import type { DayValues } from '../utils';
import {
  useMonthNames,
  useNextMonthMessage,
  usePreviousMonthMessage,
  useWeekdayNames,
} from '../utils';
import { SCPikaday } from './pikaday';
import { SCtheme } from './theme';

export { SCtheme } from './theme';
export { SCPikaday } from './pikaday';
interface DatePickerProps {
  defaultDate?: Date;
  firstDay?: DayValues;
  hideInput?: boolean;
  inputProps?: Partial<InputProps>;
  maxDate?: Date;
  minDate?: Date;
  onChange?: (date: Date) => void;
  pickWholeWeek?: boolean;
  trigger?: HTMLElement;
}

export const DatePicker: FC<DatePickerProps> = (props: DatePickerProps) => {
  const { getLanguageCode } = useIntl();
  const [loadingDependencies, setLoadingDependencies] = useState<boolean>(true);
  const i18n = {
    months: useMonthNames(),
    nextMonth: useNextMonthMessage(),
    previousMonth: usePreviousMonthMessage(),
    weekdays: useWeekdayNames('long'),
    weekdaysShort: useWeekdayNames('short'),
  };

  const prevProps = usePrevious(props);

  const picker = useRef<Pikaday | null>(null);
  const textInput = useRef<HTMLInputElement | null>(null);
  const containerElement = useRef<HTMLDivElement | null>(null);

  const handleSelect = (date: Date) => {
    props.onChange?.(date);
  };

  useEffectOnce(() => {
    const momentLocale = getLanguageCode();
    moment.locale(momentLocale || 'en');

    picker.current = new Pikaday({
      container: containerElement.current,
      defaultDate: props.defaultDate,
      field: textInput.current,
      // Pikaday uses default on null, not undefined
      firstDay: props.firstDay || null,
      format: 'LL',
      i18n,
      maxDate: props.maxDate,
      minDate: props.minDate,
      onSelect: handleSelect,
      parse: (dateString: string, format?) =>
        moment(dateString, format).toDate(),
      pickWholeWeek: props.pickWholeWeek,
      setDefaultDate: true,
      theme: 'inline',
      toString: (date, format?) => moment(date).format(format),
      trigger: props.trigger,
      // Pikaday uses default on null for firstDay but the type doesn't match(type is undefined));
    } as Pikaday.PikadayOptions);
    setLoadingDependencies(false);
  });

  useEffect(() => {
    if (!picker.current) {
      return;
    }
    if (
      dateDidChange(prevProps.minDate, props.minDate) ||
      dateDidChange(prevProps.maxDate, props.maxDate)
    ) {
      if (props.minDate) {
        picker.current.setMinDate(props.minDate);
      }
      if (props.maxDate) {
        picker.current.setMaxDate(props.maxDate);
      }
    }
  });

  useEffect(() => {
    if (!picker.current) {
      return;
    }

    if (dateDidChange(prevProps.defaultDate, props.defaultDate)) {
      if (!props.defaultDate && textInput.current) {
        // Workaround for pikaday not clearing value when date set to falsey
        textInput.current.value = '';
      }
      // 2nd param = don't call onSelect
      picker.current.setDate(props.defaultDate || new Date(), true);
    }
  });

  const inputProps = props.inputProps;
  return (
    <Layout data-a-target="date-picker">
      <SCPikaday />
      <SCtheme />
      <Layout display={props.hideInput ? Display.Hide : Display.Block}>
        <Input
          refDelegate={textInput}
          type={InputType.Text}
          {...inputProps}
          readOnly={!picker || loadingDependencies}
          size={InputSize.Large}
        />
      </Layout>
      <InjectLayout>
        <div ref={containerElement} />
      </InjectLayout>
    </Layout>
  );
};

DatePicker.displayName = 'DatePicker';

function dateDidChange(prevDate: Date | undefined, newDate: Date | undefined) {
  const prevTime = prevDate ? prevDate.getTime() : null;
  const newTime = newDate ? newDate.getTime() : null;
  return prevTime !== newTime;
}
