import { DefaultTwitchIntl } from 'tachyon-intl';
import {
  createMountWrapperFactory,
  createShallowWrapperFactory,
} from 'tachyon-test-utils';
import { Display, Input, SVGAsset } from 'twitch-core-ui';
import { DatePicker } from '.';

const _getLanguageCode = DefaultTwitchIntl.getLanguageCode;
const getLanguageCode = jest.fn();
DefaultTwitchIntl.getLanguageCode = getLanguageCode;
const ENTER_KEY = 13;
const RIGHT_ARROW_KEY = 39;

describe(DatePicker, () => {
  afterAll(() => {
    DefaultTwitchIntl.getLanguageCode = _getLanguageCode;
  });

  const setupShallow = createShallowWrapperFactory(DatePicker, () => {
    return {
      defaultDate: new Date('Jan 1, 2000'),
    };
  });

  const testDiv = document.createElement('div');
  document.body.appendChild(testDiv);
  const setupMount = createMountWrapperFactory(
    DatePicker,
    () => {
      return {
        defaultDate: new Date('Jan 1, 2000'),
      };
    },
    {
      attachTo: testDiv,
    },
  );

  beforeEach(() => {
    getLanguageCode.mockReturnValue('en');
  });

  it('allows picking a date', () => {
    const { wrapper } = setupMount();

    const nativeInput = document.querySelector('input') as HTMLInputElement;
    expect(nativeInput.value).toBe('January 1, 2000');
    nativeInput.dispatchEvent(new Event('click'));
    const pika = document.querySelector('.pika-single') as HTMLDivElement;
    expect(pika).toBeDefined();
    // fire keydown to move to next day, i couldn't get mousedown events to work to click days...
    // at least keyboard events fire.
    // tslint:disable-next-line no-any
    const keyEvent = new Event('keydown', { bubbles: true }) as any;
    keyEvent.keyCode = RIGHT_ARROW_KEY;
    pika.dispatchEvent(keyEvent);

    // select the focused day
    keyEvent.keyCode = ENTER_KEY;
    pika.dispatchEvent(keyEvent);
    expect(nativeInput.value).toBe('January 2, 2000');

    wrapper.detach();
  });

  it('allows date range to be set', () => {
    const { wrapper } = setupMount({
      defaultDate: new Date('Jan 10, 2000'),
      maxDate: new Date('Jan 15, 2000'),
      minDate: new Date('Jan 5, 2000'),
    });

    const nativeInput = document.querySelector('input') as HTMLInputElement;
    nativeInput.dispatchEvent(new Event('click'));

    let day = document.querySelector(
      '[data-day="4"]',
    ) as HTMLTableDataCellElement;
    expect(day.classList.contains('is-disabled')).toBe(true);
    day = document.querySelector('[data-day="16"]') as HTMLTableDataCellElement;
    expect(day.classList.contains('is-disabled')).toBe(true);
    day = document.querySelector('[data-day="10"]') as HTMLTableDataCellElement;

    expect(day.classList.contains('is-disabled')).toBe(false);
    expect(day.classList.contains('is-selected')).toBe(true);
    wrapper.detach();
  });

  it('can pass custom input properties', () => {
    const { wrapper } = setupShallow({
      inputProps: {
        disabled: true,
        icon: SVGAsset.Events,
        placeholder: 'test',
      },
    });
    const input = wrapper.find(Input);
    expect(input).toHaveProp('icon', SVGAsset.Events);
    expect(input).toHaveProp('disabled', true);
    expect(input).toHaveProp('placeholder', 'test');
  });

  it('updates date when defaultDate changes', () => {
    const props1 = { defaultDate: new Date('Jan 20, 2000') };
    const props2 = { defaultDate: new Date('Jan 21, 2000') };
    const { wrapper } = setupMount(props1);

    const nativeInput = document.querySelector('input') as HTMLInputElement;
    expect(nativeInput.value).toBe('January 20, 2000');
    wrapper.setProps(props2);
    wrapper.update();
    expect(nativeInput.value).toBe('January 21, 2000');

    wrapper.detach();
  });

  it("doesn't break when updating defaultDate before pikaday is loaded", () => {
    const props1 = { defaultDate: new Date('Jan 20, 2000') };
    const props2 = { defaultDate: new Date('Jan 21, 2000') };
    const { wrapper } = setupMount(props1);
    wrapper.setProps(props2);

    const nativeInput = document.querySelector('input') as HTMLInputElement;

    expect(nativeInput.value).toBe('January 21, 2000');

    wrapper.detach();
  });

  it("correctly hides Input element if 'hideInput' prop is truthy", () => {
    const { wrapper } = setupShallow({
      defaultDate: new Date('Jan 20, 2000'),
      hideInput: true,
    });

    expect(wrapper.find(Input).parent().props().display).toEqual(Display.Hide);
  });

  it('defaults to en-US locale if an invalid locale is set', () => {
    getLanguageCode.mockReturnValue('invalid');
    const { wrapper } = setupMount({
      defaultDate: new Date('Jan 20, 2000'),
    });

    const nativeInput = document.querySelector('input') as HTMLInputElement;
    expect(nativeInput.value).toBe('January 20, 2000');

    wrapper.detach();
  });

  it('formats dates in the correct locale', () => {
    getLanguageCode.mockReturnValue('ja-JP');
    const { wrapper } = setupMount({
      defaultDate: new Date('Jan 20, 2000'),
    });

    const nativeInput = document.querySelector('input') as HTMLInputElement;
    expect(nativeInput.value).toBe('2000年1月20日');

    wrapper.detach();
  });
});
