import { differenceInDays } from 'date-fns';
import type { FC } from 'react';
import { useCallback } from 'react';
import { useIntl } from 'tachyon-intl';
import {
  AlignItems,
  Background,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Interactable,
  InteractableType,
  Layout,
  Position,
  TextAlign,
  ZIndex,
} from 'twitch-core-ui';
import { DateRangePicker } from '../date-range-picker';
import type { PresetInterval } from '../date-range-picker-presets';
import { PresetID } from '../date-range-picker-presets';
import { Disclosure } from '../disclosure';
import type { Interval } from '../models';
import { shiftInterval } from '../models';
import { ShuffleWrapProps } from '../shuffle-wrap';
import { isInQueryableRange } from './utils';

export interface DateRangeBarProps {
  interval: Interval;
  onDateSelect: ({ end, id, start }: PresetInterval) => void;
}

export const TIME_RANGE_SELECTOR = { 'data-test-selector': 'time-range' };
export const TOTAL_DAYS_SELECTOR = { 'data-test-selector': 'total-days' };
export const DATE_PICKER_TOGGLE_SELECTOR = {
  'data-test-selector': 'date-picker-toggle',
};

/**
 * This component renders the UI and copy for the vertical NavBar component
 * that controls the user's selected date range.
 */
export const DateRangeBar: FC<DateRangeBarProps> = ({
  interval,
  onDateSelect,
}) => {
  const { formatDate, formatMessage } = useIntl();

  const control = useCallback(
    ({ toggle }) => {
      function formatDateRange(dInterval: Interval) {
        const numDays = differenceInDays(dInterval.end, dInterval.start) + 1;

        if (numDays === 1) {
          return formatDate(dInterval.start, 'medium');
        }

        return formatMessage(
          '{start, date, medium} – {end, date, medium}',
          { end: dInterval.end, start: dInterval.start },
          'AnalyticsNavBar',
        );
      }

      function formatDaysInRange(dInterval: Interval) {
        const numDays = differenceInDays(dInterval.end, dInterval.start) + 1;
        // {numDays, plural, one {# day} other {# days}}',
        return formatMessage(
          '{numDays, plural, one {# day} other {# days}}',
          { numDays },
          'AnalyticsNavBar',
        );
      }

      const handleClickNext = (): void => {
        pathWithShiftedInterval(1);
      };
      const handleClickPrevious = (): void => {
        pathWithShiftedInterval(-1);
      };

      function pathWithShiftedInterval(direction: number) {
        const nextInterval = shiftInterval(interval, direction);
        if (isInQueryableRange(nextInterval)) {
          onDateSelect({
            end: nextInterval.end,
            id: PresetID.Custom,
            start: nextInterval.start,
          });
        }
      }
      return (
        <Layout padding={{ bottom: 2 }}>
          <ShuffleWrapProps
            onClickNext={handleClickNext}
            onClickPrevious={handleClickPrevious}
          >
            <Layout
              background={Background.Base}
              display={Display.Flex}
              fullWidth
            >
              <Interactable
                {...DATE_PICKER_TOGGLE_SELECTOR}
                onClick={toggle}
                variant={InteractableType.Alpha}
              >
                <Layout textAlign={TextAlign.Center}>
                  <CoreText
                    color={Color.Link}
                    {...TIME_RANGE_SELECTOR}
                    fontSize={FontSize.Size4}
                  >
                    {formatDateRange(interval)}
                  </CoreText>
                  <CoreText
                    {...TOTAL_DAYS_SELECTOR}
                    color={Color.Alt2}
                    fontSize={FontSize.Size6}
                  >
                    {formatDaysInRange(interval)}
                  </CoreText>
                </Layout>
              </Interactable>
            </Layout>
          </ShuffleWrapProps>
        </Layout>
      );
    },
    [formatDate, formatMessage, interval, onDateSelect],
  );

  return (
    <Layout position={Position.Relative} zIndex={ZIndex.Above}>
      <Disclosure control={control}>
        {({ close }) => (
          <Layout
            alignItems={AlignItems.Center}
            attachLeft
            attachRight
            attachTop
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            margin={{ top: 5 }}
            position={Position.Absolute}
            zIndex={ZIndex.Above}
          >
            <Layout
              attachTop
              background={Background.Base}
              display={Display.Flex}
              elevation={4}
              margin={{ top: 0.5 }}
              position={Position.Relative}
              zIndex={ZIndex.Above}
            >
              <Layout padding={{ bottom: 1, left: 2, right: 2, top: 2 }}>
                <DateRangePicker
                  interval={interval}
                  onCommit={(date: PresetInterval) => {
                    close();
                    onDateSelect(date);
                  }}
                  onDismiss={close}
                />
              </Layout>
            </Layout>
          </Layout>
        )}
      </Disclosure>
    </Layout>
  );
};

DateRangeBar.displayName = 'DateRangeBar';
