import {
  endOfDay,
  endOfMonth,
  startOfDay,
  startOfMonth,
  subDays,
  subMonths,
} from 'date-fns';
import type { FC } from 'react';
import { useMemo } from 'react';
import { useIntl } from 'tachyon-intl';
import {
  BorderRadius,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Interactable,
  Layout,
  TextAlign,
  WhiteSpace,
} from 'twitch-core-ui';
import type { Interval } from '../models';
import { isEqualInterval } from '../models';

interface DateRangePickerPresetsProp {
  interval: Interval;
  now?: Date;
  onChange: (interval: PresetInterval) => void;
}

/* eslint-disable typescript-sort-keys/string-enum */
export enum PresetID {
  Custom = 'custom',
  Default = 'default',
  Last7Days = 'last_7_days',
  Last30Days = 'last_30_days',
  CurrentMonth = 'current_month',
  oneMonthAgo = 't_minus_1_month',
  twoMonthsAgo = 't_minus_2_month',
}
/* eslint-enable typescript-sort-keys/string-enum */

export interface PresetInterval extends Interval {
  id: PresetID;
}

export const DateRangePickerPresets: FC<DateRangePickerPresetsProp> = (
  props: DateRangePickerPresetsProp,
) => {
  const { formatDate, formatMessage } = useIntl();
  const getPresets = useMemo(() => {
    const now = props.now || new Date();
    const endOfToday = endOfDay(now);
    const startOfToday = startOfDay(now);
    const startOfThisMonth = startOfMonth(now);

    return [
      {
        interval: {
          end: endOfToday,
          id: PresetID.Last7Days,
          start: subDays(startOfToday, 6),
        },
        message: formatMessage('Last 7 Days', 'DevSiteDateRangePickerPresets'),
      },
      {
        interval: {
          end: endOfToday,
          id: PresetID.Last30Days,
          start: subDays(startOfToday, 29),
        },
        message: formatMessage('Last 30 Days', 'DevSiteDateRangePickerPresets'),
      },
      {
        interval: {
          end: endOfToday,
          id: PresetID.CurrentMonth,
          start: startOfThisMonth,
        },
        message: formatDate(startOfThisMonth, {
          month: 'long',
          year: 'numeric',
        }),
      },
      {
        interval: {
          end: endOfMonth(subMonths(now, 1)),
          id: PresetID.oneMonthAgo,
          start: startOfMonth(subMonths(now, 1)),
        },
        message: formatDate(subMonths(endOfToday, 1), {
          month: 'long',
          year: 'numeric',
        }),
      },
      {
        interval: {
          end: endOfMonth(subMonths(now, 2)),
          id: PresetID.twoMonthsAgo,
          start: startOfMonth(subMonths(now, 2)),
        },
        message: formatDate(subMonths(endOfToday, 2), {
          month: 'long',
          year: 'numeric',
        }),
      },
    ];
  }, [formatDate, formatMessage, props.now]);

  const presets = getPresets;

  function handleClickFor(interval: PresetInterval): () => void {
    return () => props.onChange(interval);
  }

  return (
    <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
      {presets.map((preset, i) => (
        <Layout color={Color.Link} key={i} margin={{ bottom: 0.5 }}>
          <Interactable
            borderRadius={BorderRadius.Small}
            onClick={handleClickFor(preset.interval)}
            selected={isEqualInterval(props.interval, preset.interval)}
          >
            <Layout padding={1} textAlign={TextAlign.Center}>
              <CoreText
                fontSize={FontSize.Size5}
                whiteSpace={WhiteSpace.NoWrap}
              >
                {preset.message}
              </CoreText>
            </Layout>
          </Interactable>
        </Layout>
      ))}
    </Layout>
  );
};
DateRangePickerPresets.displayName = 'DateRangePickerPresets';
