import {
  endOfDay,
  parse as parseDate,
  startOfDay,
  startOfMonth,
} from 'date-fns';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { Interactable } from 'twitch-core-ui';
import { DateRangePickerPresets } from '.';

describe(DateRangePickerPresets, () => {
  const now = parseDate('2000-01-01');
  const setup = createShallowWrapperFactory(DateRangePickerPresets, () => ({
    interval: { end: now, start: now },
    now,
    onChange: jest.fn(),
  }));

  it('showing a selected button', () => {
    const { wrapper } = setup();
    let buttons = wrapper.find(Interactable);

    expect(buttons.at(0)).toHaveProp('selected', false);
    expect(buttons.at(1)).toHaveProp('selected', false);
    expect(buttons.at(2)).toHaveProp('selected', false);
    expect(buttons.at(3)).toHaveProp('selected', false);
    expect(buttons.at(4)).toHaveProp('selected', false);

    const currentMonthInterval = {
      end: endOfDay(now),
      start: startOfMonth(now),
    };

    wrapper.setProps({ interval: currentMonthInterval });
    buttons = wrapper.find(Interactable);

    expect(buttons.at(0)).toHaveProp('selected', false);
    expect(buttons.at(1)).toHaveProp('selected', false);
    expect(buttons.at(2)).toHaveProp('selected', true);
    expect(buttons.at(3)).toHaveProp('selected', false);
    expect(buttons.at(4)).toHaveProp('selected', false);
  });

  it('calling onChange with interval', () => {
    const { props, wrapper } = setup();
    const buttons = wrapper.find(Interactable);

    buttons.forEach((b) => b.simulate('click'));
    const calls = (props.onChange as jest.Mock).mock.calls;

    // Last 7 days
    expect(calls[0]).toEqual([
      {
        end: endOfDay('2000-01-01'),
        id: 'last_7_days',
        start: startOfDay('1999-12-26'),
      },
    ]);

    // Last 30 days
    expect(calls[1]).toEqual([
      {
        end: endOfDay('2000-01-01'),
        id: 'last_30_days',
        start: startOfDay('1999-12-03'),
      },
    ]);

    // Current month
    expect(calls[2]).toEqual([
      {
        end: endOfDay('2000-01-01'),
        id: 'current_month',
        start: startOfDay('2000-01-01'),
      },
    ]);

    // Previous month
    expect(calls[3]).toEqual([
      {
        end: endOfDay('1999-12-31'),
        id: 't_minus_1_month',
        start: startOfDay('1999-12-01'),
      },
    ]);

    // Two months ago
    expect(calls[4]).toEqual([
      {
        end: endOfDay('1999-11-30'),
        id: 't_minus_2_month',
        start: startOfDay('1999-11-01'),
      },
    ]);
  });
});
