import { endOfDay } from 'date-fns';
import type { FC } from 'react';
import { useState } from 'react';
import { useIntl } from 'tachyon-intl';
import {
  Button,
  ButtonType,
  Color,
  CoreText,
  Display,
  FlexWrap,
  FontSize,
  JustifyContent,
  Layout,
  TextAlign,
} from 'twitch-core-ui';
import { DatePickerLite as DatePicker } from '../date-picker-lite';
import type { PresetInterval } from '../date-range-picker-presets';
import { DateRangePickerPresets, PresetID } from '../date-range-picker-presets';
import type { Interval } from '../models';
import { updateInterval } from './utils';

interface DateRangePickerProps {
  interval: Interval;
  onCommit: (interval: PresetInterval) => void;
  onDismiss: () => void;
}

export const COMMIT_BUTTON_SELECTOR = { 'data-test-selector': 'commit-button' };
export const CANCEL_BUTTON_SELECTOR = { 'data-test-selector': 'cancel-button' };

// We only have analytics data back to 2017-01-01.
const startOfAnalytics = new Date('2017-01-01');

/**
 * This component renders other components that allow users to select a time
 * range to view. Rules for which days to enable in the date pickers are
 * configured here.
 *
 * `state` temporarily stores the interval as the user interacts with the
 * selection UI. Changes to the dates are passed through through the
 * `updateInterval` policy function which may adjust the interval before the
 * state changes. The user then commits or discards the state when they are
 * done selecting.
 */
export const DateRangePicker: FC<DateRangePickerProps> = (
  props: DateRangePickerProps,
) => {
  const [interval, setInterval] = useState<Interval>(props.interval);
  const { formatMessage } = useIntl();
  const handleStartChange = (start: Date) => {
    const updatedInterval = updateInterval(interval, { start });
    setInterval(updatedInterval);
  };

  const handleEndChange = (end: Date) => {
    const updatedInterval = updateInterval(interval, {
      end: endOfDay(end),
    });
    setInterval(updatedInterval);
  };

  const handleUpdateClick = () => {
    props.onCommit({ ...interval, id: PresetID.Custom });
  };

  const now = new Date();

  return (
    <div>
      <Layout
        display={Display.Flex}
        flexWrap={FlexWrap.NoWrap}
        margin={{ bottom: 2 }}
      >
        <Layout padding={{ right: 2 }} textAlign={TextAlign.Center}>
          <CoreText color={Color.Alt2} fontSize={FontSize.Size4}>
            {formatMessage('Start date', 'DevSiteDateRangePicker')}
          </CoreText>
          <DatePicker
            date={interval.start}
            endRange={interval.end}
            maxDate={now}
            minDate={startOfAnalytics}
            onChange={handleStartChange}
            startRange={interval.start}
          />
        </Layout>
        <Layout padding={{ right: 2 }} textAlign={TextAlign.Center}>
          <CoreText color={Color.Alt2} fontSize={FontSize.Size4}>
            {formatMessage('End date', 'DevSiteDateRangePicker')}
          </CoreText>
          <DatePicker
            date={interval.end}
            endRange={interval.end}
            maxDate={now}
            minDate={interval.start}
            onChange={handleEndChange}
            startRange={interval.start}
          />
        </Layout>
        <Layout borderLeft padding={{ left: 2, top: 0.5 }}>
          <DateRangePickerPresets
            interval={interval}
            onChange={props.onCommit}
          />
        </Layout>
      </Layout>
      <Layout display={Display.Flex} justifyContent={JustifyContent.Center}>
        <Layout margin={{ right: 2 }}>
          <Button {...COMMIT_BUTTON_SELECTOR} onClick={handleUpdateClick}>
            {formatMessage('Update', 'DevSiteDateRangePicker')}
          </Button>
        </Layout>
        <Button
          {...CANCEL_BUTTON_SELECTOR}
          onClick={props.onDismiss}
          variant={ButtonType.Text}
        >
          {formatMessage('Clear', 'DevSiteDateRangePicker')}
        </Button>
      </Layout>
    </div>
  );
};
DateRangePicker.displayName = 'DateRangePicker';
