import { format, parse } from 'date-fns';
import type { Interval } from '../../../models';
import { updateInterval } from '.';

const today = parse('2001-01-01');
const todayInterval = { end: today, start: today };

describe('picking a valid date range', () => {
  it('picking a past start date', () => {
    const i = updateInterval(
      todayInterval,
      { start: parse('2000-12-31') },
      today,
    );

    expectIntervalMatch(i, { end: today, start: parse('2000-12-31') });
  });

  it('picking a future end date', () => {
    const i = updateInterval(
      todayInterval,
      { end: parse('2001-01-02') },
      today,
    );

    expectIntervalMatch(i, { end: parse('2001-01-02'), start: today });
  });

  it('picking the same interval', () => {
    const i = updateInterval(
      todayInterval,
      { end: today, start: today },
      today,
    );

    expectIntervalMatch(i, { end: today, start: today });
  });
});

it('picking a start date that is after the end date - one day range', () => {
  const i = updateInterval(
    { end: parse('2000-12-31'), start: parse('2000-12-31') },
    { start: parse('2001-01-01') },
    today,
  );

  expectIntervalMatch(i, {
    end: parse('2001-01-01'),
    start: parse('2001-01-01'),
  });
});

it('picking a start date that is after the end date - two day range', () => {
  const i = updateInterval(
    { end: parse('2000-12-30'), start: parse('2000-12-29') },
    { start: parse('2000-12-31') },
    today,
  );

  expectIntervalMatch(i, {
    end: parse('2001-01-01'),
    start: parse('2000-12-31'),
  });
});

it('picking a start date that is after the end date - compressing the range to end today', () => {
  const i = updateInterval(
    { end: parse('2000-12-31'), start: parse('2000-12-30') },
    { start: parse('2001-01-01') },
    today,
  );

  expectIntervalMatch(i, {
    end: parse('2001-01-01'),
    start: parse('2001-01-01'),
  });
});

it('picking an end date that is before the start date - one day range', () => {
  const i = updateInterval(todayInterval, { end: parse('2000-12-31') }, today);

  expectIntervalMatch(i, {
    end: parse('2000-12-31'),
    start: parse('2000-12-31'),
  });
});

it('picking an invalid range where both start and end have changed', () => {
  const i = updateInterval(
    { end: parse('2000-12-30'), start: parse('2000-12-29') },
    { end: parse('2000-12-29'), start: parse('2000-12-30') },
    today,
  );

  expectIntervalMatch(i, {
    end: parse('2000-12-30'),
    start: parse('2000-12-29'),
  });
});

function expectIntervalMatch(actual: Interval, expected: Interval) {
  expect(serializeInterval(actual)).toEqual(serializeInterval(expected));
}

function serializeInterval(interval: Interval) {
  return `${formatDate(interval.start)} - ${formatDate(interval.end)}`;
}

function formatDate(date: Date) {
  return format(date, 'YYYY-MM-DD');
}
