import {
  addHours,
  endOfDay,
  isPast,
  isToday,
  isValid,
  setMinutes,
} from 'date-fns';
import { Component } from 'react';
import { Display, FlexDirection, FormGroup, Layout } from 'twitch-core-ui';
import { DatePicker } from '../date-picker';
import type { Time, TimeOption } from '../models';
import { RelativeTime } from '../models';
import { TimePicker } from '../time-picker';
import { isValid as isValidTime, toUTCDate } from '../utils';

export interface DateTimePickerProps {
  allowPast?: boolean;
  customTimeOptions?: TimeOption[];
  dateErrorMessage: string;
  dateLabel: string;
  initialDate?: Date;
  onChange?: (dateTime: Date) => void;
  removeNow?: boolean;
  timeErrorMessage: string;
  timeLabel: string;
  useUTC?: boolean;
}

// exporting just for tests
export function generateTimeStrings(
  startDate: Date | undefined = undefined,
): string[] {
  const times = [];
  const minutes = ['00', '30'];
  const startHour = startDate ? startDate.getHours() : 0;

  for (let hour = startHour; hour < 24; ++hour) {
    for (const minute of minutes) {
      if (hour > 11) {
        times.push(`${hour === 12 ? 12 : hour - 12}:${minute}pm`);
      } else {
        times.push(`${hour === 0 ? 12 : hour}:${minute}am`);
      }
    }
  }

  if (startDate && startDate.getMinutes() >= 30) {
    times.splice(0, 2);
  } else if (startDate) {
    times.splice(0, 1);
  }
  return times;
}

export interface State {
  computedDateTime: Date;
  date: Date;
  invalidDate?: boolean;
  invalidTime?: boolean;
  time: Time | null;
  timeValue?: string | undefined;
}

const INVALID_DATE = new Date('INVALID DATE');

export class DateTimePicker extends Component<DateTimePickerProps, State> {
  displayName = 'DateTimePicker';
  constructor(props: DateTimePickerProps) {
    super(props);
    const defaultDate = this.props.initialDate || this.getDefaultDate();
    this.state = {
      computedDateTime: defaultDate,
      date: defaultDate,
      time: this.dateToTime(defaultDate),
    };
  }

  public override render(): JSX.Element {
    return (
      <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
        <Layout margin={{ bottom: 1 }} padding={{ right: 2 }}>
          <FormGroup
            error={!!this.state.invalidDate}
            errorMessage={this.props.dateErrorMessage}
            id="event-start-date"
            label={this.props.dateLabel}
          >
            <DatePicker
              defaultDate={this.state.date}
              inputProps={{ readOnly: true }}
              onChange={this.onChangeDate}
            />
          </FormGroup>
        </Layout>
        <FormGroup
          error={!!this.state.invalidTime}
          errorMessage={this.props.timeErrorMessage}
          id="event-start-time"
          label={`${this.props.timeLabel} (${this.getLocaleShortName()})`}
        >
          <TimePicker
            customTimeOptions={this.props.customTimeOptions}
            initialTime={this.state.time}
            onChange={this.onChangeTime}
          />
        </FormGroup>
      </Layout>
    );
  }

  public onChangeTime = (newTime: Time | null, value: string): void => {
    if (value === RelativeTime.Now) {
      const now = this.getNow();
      this.setDateTime(now, this.dateToTime(now), value);
    } else {
      this.setTime(newTime, value);
    }
  };

  private notifyOnChange(newState: Partial<State>): void {
    if (
      this.props.onChange &&
      (this.state.computedDateTime !== newState.computedDateTime ||
        this.state.timeValue !== newState.timeValue)
    ) {
      const timeValue = newState.timeValue
        ? newState.timeValue
        : this.state.timeValue;
      const computedDateTime = newState.computedDateTime
        ? newState.computedDateTime
        : this.state.computedDateTime;
      this.props.onChange(
        timeValue === RelativeTime.Now ? this.getNow() : computedDateTime,
      );
    }
  }

  private onChangeDate = (newDate: Date): void => {
    // If date isn't today but timeValue is still Now, reset time.
    if (!isToday(newDate) && this.state.timeValue === RelativeTime.Now) {
      this.setTime(null, '');
    }
    this.setDate(newDate);
  };

  private setDateTime(date: Date, time: Time, timeValue?: string): void {
    const computedDateTime = this.computeDateTime(date, time);
    const newState = {
      computedDateTime,
      date,
      invalidDate:
        (!this.props.allowPast && isPast(endOfDay(date))) || !isValid(date),
      invalidTime: !isValidTime(time),
      time,
      timeValue,
    };

    this.notifyOnChange(newState);
    this.setState(newState);
  }

  private setTime(time: Time | null, timeValue?: string): void {
    const { date } = this.state;
    const computedDateTime = this.computeDateTime(date, time);
    const newState = {
      computedDateTime,
      invalidDate: !this.props.allowPast && isPast(endOfDay(date)),
      invalidTime:
        (!this.props.allowPast && isPast(computedDateTime)) ||
        !isValid(computedDateTime),
      time,
      timeValue,
    };

    this.notifyOnChange(newState);
    this.setState(newState);
  }

  private setDate(date: Date): void {
    const computedDateTime = this.computeDateTime(date, this.state.time);

    const newState = {
      computedDateTime,
      date,
      invalidDate:
        (!this.props.allowPast && isPast(endOfDay(date))) ||
        !isValid(computedDateTime),
      invalidTime: !this.props.allowPast && isPast(computedDateTime),
    };
    this.notifyOnChange(newState);
    this.setState(newState);
  }

  private computeDateTime(date: Date | undefined, time: Time | null): Date {
    if (!date || !time || !isValid(date) || !isValidTime(time)) {
      return INVALID_DATE;
    }
    date.setHours(time.hours);
    date.setMinutes(time.minutes);
    date.setSeconds(0);
    return date;
  }

  private dateToTime(date: Date): Time {
    return {
      hours: date.getHours(),
      minutes: date.getMinutes(),
    };
  }

  private getLocaleShortName = (): string => {
    return this.props.useUTC
      ? 'UTC'
      : new Date()
          .toLocaleTimeString('en-us', { timeZoneName: 'short' })
          .split(' ')[2];
  };

  private getNow = (): Date => {
    return this.props.useUTC ? toUTCDate(new Date()) : new Date();
  };

  private getDefaultDate(): Date {
    return setMinutes(addHours(this.getNow(), 3), 0);
  }
}
