import { isEqual } from 'date-fns';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { RelativeTime } from '..';
import { DateTimePicker, generateTimeStrings } from '.';

describe(DateTimePicker, () => {
  const setupShallow = createShallowWrapperFactory(DateTimePicker, () => ({
    dateErrorMessage: 'Start Date is invalid',
    dateLabel: 'Start Date',
    initialDate: new Date('April 20, 2018 4:44 UTC'),
    onChange: jest.fn(),
    timeErrorMessage: 'Start Time is invalid',
    timeLabel: 'Start Time',
  }));
  it('allows inputting a time to get a new Date', () => {
    const { props, wrapper } = setupShallow();
    const testTime = { hours: 16, minutes: 20, seconds: 0 };
    const instance = wrapper.instance() as DateTimePicker;
    expect(instance).toBeInstanceOf(DateTimePicker);
    instance.onChangeTime(testTime, '16:20');
    wrapper.update();
    expect(props.onChange).toHaveBeenCalledTimes(1);
    expect(props.onChange).toHaveBeenCalledWith(expect.any(Date));
    const calledDate = (props.onChange as jest.Mock).mock.calls[0][0] as Date;
    expect(calledDate.getHours()).toBe(16);
    expect(calledDate.getMinutes()).toBe(20);
    expect(instance.state.time).toBeDefined();
    expect(isEqual(calledDate, instance.state.computedDateTime)).toBe(true);
  });

  it('creates an invalid Date when time input is invalid', () => {
    const { props, wrapper } = setupShallow();
    const invalidTime = null;
    const instance = wrapper.instance() as DateTimePicker;
    expect(instance).toBeInstanceOf(DateTimePicker);
    instance.onChangeTime(invalidTime, '');
    wrapper.update();
    expect(props.onChange).toHaveBeenCalledTimes(1);
    expect(props.onChange).toHaveBeenCalledWith(expect.any(Date));
    const calledDate = (props.onChange as jest.Mock).mock.calls[0][0] as Date;
    expect(isNaN(calledDate.getHours())).toBe(true);
    expect(isNaN(calledDate.getMinutes())).toBe(true);
  });

  it("sets date correctly when 'Now' is submitted for as time", () => {
    const { props, wrapper } = setupShallow();
    const nowDate = new Date();
    const now = { hours: nowDate.getHours(), minutes: nowDate.getMinutes() };
    const instance = wrapper.instance() as DateTimePicker;
    expect(instance).toBeInstanceOf(DateTimePicker);
    instance.onChangeTime(now, RelativeTime.Now);
    wrapper.update();
    expect(props.onChange).toHaveBeenCalledTimes(1);
    expect(props.onChange).toHaveBeenCalledWith(expect.any(Date));
    const calledDate = (props.onChange as jest.Mock).mock.calls[0][0] as Date;
    expect(calledDate.getHours()).toBe(now.hours);
    expect(calledDate.getMinutes()).toBe(now.minutes);
    expect(instance.state.timeValue).toBe(RelativeTime.Now);
  });
});

describe('generateTimeStrings', () => {
  it('should generate all times of day in 30 minute intervals', () => {
    const generatedTimes = generateTimeStrings();
    expect(generatedTimes).toHaveLength(48);
    expect(generatedTimes[0]).toBe('12:00am');
    expect(generatedTimes[47]).toBe('11:30pm');
  });

  it('should generate all times of day in the future before a :30', () => {
    const generatedTimes = generateTimeStrings(new Date('April 20, 2018 3:00'));
    expect(generatedTimes).toHaveLength(41);
    expect(generatedTimes[0]).toBe('3:30am');
    expect(generatedTimes[40]).toBe('11:30pm');
  });

  it('should generate all times of day in the future after a :30', () => {
    const generatedTimes = generateTimeStrings(new Date('April 20, 2018 3:31'));
    expect(generatedTimes).toHaveLength(40);
    expect(generatedTimes[0]).toBe('4:00am');
    expect(generatedTimes[39]).toBe('11:30pm');
  });
});
