import {
  addDays,
  differenceInDays,
  differenceInMinutes,
  endOfDay,
  isEqual,
  isValid,
  subDays,
} from 'date-fns';

export interface Interval {
  end: Date;
  start: Date;
}

export function isEqualInterval(a: Interval, b: Interval): boolean {
  return isEqual(a.start, b.start) && isEqual(a.end, b.end);
}

export function isSameLength(a: Interval, b: Interval): boolean {
  return (
    a.end.getTime() - a.start.getTime() === b.end.getTime() - b.start.getTime()
  );
}

export function durationInMinutes(interval: Interval): number {
  return differenceInMinutes(interval.end, interval.start);
}

export function durationInMilliseconds(interval: Interval): number {
  return interval.end.getTime() - interval.start.getTime();
}

export function lastNDays(n: number, now = new Date()): Interval {
  const endOfToday = endOfDay(now);

  return { end: endOfToday, start: subDays(endOfToday, n) };
}

/**
 * Given an interval, create a new lookback interval that is shifted forward or
 * backward.
 */
export function shiftInterval(interval: Interval, direction: number): Interval {
  const dayDiff = differenceInDays(interval.end, interval.start);
  const shiftDays = (dayDiff + 1) * direction;

  return {
    end: addDays(interval.end, shiftDays),
    start: addDays(interval.start, shiftDays),
  };
}

export function intervalBeginsBefore(
  interval: Interval,
  otherInterval: Interval,
): boolean {
  return otherInterval.start.getTime() < interval.start.getTime();
}

export function isIntervalValid(interval: Interval): boolean {
  return (
    isValid(interval.start) &&
    isValid(interval.end) &&
    interval.start.getTime() <= interval.end.getTime()
  );
}
