import type { FC, MouseEvent, ReactNode } from 'react';
import styled from 'styled-components';
import {
  AlignItems,
  Background,
  Color,
  Display,
  FlexWrap,
  InjectLayout,
  Interactable,
  InteractableType,
  JustifyContent,
  Layout,
  SVG,
  SVGAsset,
  SVGType,
  getDataProps,
} from 'twitch-core-ui';

export interface ShuffleWrapProps {
  children: ReactNode;
  onClickNext: () => void;
  onClickPrevious: () => void;
}

const ScNavBar = styled.div`
  height: 5rem;
  user-select: none;
`;

/**
 * Used to wrap a component on both sides with buttons to shuffle.
 * Usually accompanies time analytics to scroll through dates
 */
export const ShuffleWrapProps: FC<ShuffleWrapProps> = ({
  children,
  onClickNext,
  onClickPrevious,
}) => {
  return (
    <InjectLayout
      alignItems={AlignItems.Stretch}
      background={Background.Base}
      className="nav-bar-layout__height"
      display={Display.Flex}
      elevation={1}
      flexWrap={FlexWrap.NoWrap}
    >
      <ScNavBar>
        <ShuffleWrapSquareComponent onClick={onClickPrevious}>
          <SVG asset={SVGAsset.GlyphArrLeft} type={SVGType.Inherit} />
        </ShuffleWrapSquareComponent>

        <Layout
          alignItems={AlignItems.Stretch}
          borderLeft
          borderRight
          display={Display.Flex}
          flexGrow={1}
          flexWrap={FlexWrap.NoWrap}
          justifyContent={JustifyContent.Center}
        >
          {children}
        </Layout>

        <ShuffleWrapSquareComponent onClick={onClickNext}>
          <SVG asset={SVGAsset.GlyphArrRight} type={SVGType.Inherit} />
        </ShuffleWrapSquareComponent>
      </ScNavBar>
    </InjectLayout>
  );
};
ShuffleWrapProps.displayName = 'ShuffleWrap';

interface ShuffleWrapSquareComponentProps
  extends React.HTMLAttributes<HTMLElement> {
  onClick?: () => void;
}

const ShuffleWrapSquareComponent: FC<ShuffleWrapSquareComponentProps> = (
  props,
) => {
  const clickHandler = (_: MouseEvent<HTMLAnchorElement>) => {
    if (props.onClick) {
      props.onClick();
    }
  };

  return (
    <Layout alignItems={AlignItems.Stretch} display={Display.Flex} flexGrow={0}>
      <Layout background={Background.Base} display={Display.Flex} fullWidth>
        <Interactable
          onClick={clickHandler}
          variant={InteractableType.Alpha}
          {...getDataProps(props)}
        >
          <Layout
            alignItems={AlignItems.Center}
            color={Color.Link}
            display={Display.Flex}
            flexWrap={FlexWrap.NoWrap}
            fullHeight
            padding={1}
          >
            <Layout
              alignItems={AlignItems.Center}
              display={Display.Flex}
              padding={{ x: 0.5 }}
            >
              {props.children}
            </Layout>
          </Layout>
        </Interactable>
      </Layout>
    </Layout>
  );
};

ShuffleWrapSquareComponent.displayName = 'ShuffleWrapSquareComponent';
