import type { FormEvent, KeyboardEvent, MouseEvent } from 'react';
import { Component } from 'react';
import styled from 'styled-components';
import {
  AttachedBalloon,
  InjectLayout,
  Input,
  InputSize,
  InputType,
  Layout,
  Position,
} from 'twitch-core-ui';
import { ClickOutDetector } from '../click-out-detector';
import type { Time, TimeOption } from '../models';
import { TimeInteractable } from '../time-interactable';
import { DateFormats, formatTime, parseTimeStringToTime } from '../utils';

export interface TimePickerProps {
  customTimeOptions?: TimeOption[] | undefined;
  initialTime: Time | null;
  onChange: (time: Time | null, value: string) => void;
}

interface State {
  displayValue: string;
  isOpen: boolean;
  time?: Time | null;
  value: string;
}

const ScDropdown = styled.div`
  height: 20rem;
  overflow-x: none;
  overflow-y: scroll;
`;

export class TimePicker extends Component<TimePickerProps, State> {
  displayName = 'TimePicker';
  private selectedTime: HTMLDivElement | null = null;
  private timeSelector: HTMLDivElement | null = null;

  constructor(props: TimePickerProps) {
    super(props);
    const defaultValue = props.initialTime
      ? this.timeToTimeString(props.initialTime)
      : '';
    this.state = {
      displayValue: defaultValue,
      isOpen: false,
      value: defaultValue,
    };
  }

  public override componentDidUpdate(
    _: TimePickerProps,
    prevState: State,
  ): void {
    if (this.state.isOpen && !prevState.isOpen) {
      if (this.selectedTime && this.timeSelector) {
        this.timeSelector.scrollTop = this.selectedTime.offsetTop;
      }
    }
  }

  public override render(): JSX.Element {
    const timeOptions = this.props.customTimeOptions
      ? this.props.customTimeOptions.concat(this.generateTimeOptions())
      : this.generateTimeOptions();
    const options = timeOptions.map((option: TimeOption) => (
      <TimeInteractable
        {...option}
        key={option.value}
        onClick={this.onClickTime}
        refDelegate={
          option.value === this.state.value
            ? this.setSelectedTimeRef
            : undefined
        }
        selected={option.value === this.state.value}
      />
    ));
    return (
      <Layout
        className="time-picker"
        data-a-target="time-pick-field"
        position={Position.Relative}
      >
        <ClickOutDetector onClickOut={this.closeDropdown}>
          <Input
            onBlur={this.onBlur}
            onChange={this.onInputChange}
            onFocus={this.onFocus}
            onKeyDown={this.onKeyDown}
            size={InputSize.Large}
            type={InputType.Text}
            value={this.state.displayValue}
          />
          <Layout className="time-picker__balloon" position={Position.Absolute}>
            <AttachedBalloon
              data-a-target="time-selector-balloon"
              offsetX="0.1rem"
              offsetY="0.1rem"
              show={this.state.isOpen}
            >
              <InjectLayout className="time-picker__dropdown">
                <ScDropdown
                  ref={(ref: HTMLDivElement) => (this.timeSelector = ref)}
                >
                  {options}
                </ScDropdown>
              </InjectLayout>
            </AttachedBalloon>
          </Layout>
        </ClickOutDetector>
      </Layout>
    );
  }

  public generateTimeStrings(
    startDate: Date | undefined = undefined,
  ): string[] {
    const times = [];
    const minutes = ['00', '30'];
    const startHour = startDate ? startDate.getHours() : 0;

    for (let hour = startHour; hour < 24; ++hour) {
      for (const minute of minutes) {
        if (hour > 11) {
          times.push(`${hour === 12 ? 12 : hour - 12}:${minute}pm`);
        } else {
          times.push(`${hour === 0 ? 12 : hour}:${minute}am`);
        }
      }
    }

    if (startDate && startDate.getMinutes() >= 30) {
      times.splice(0, 2);
    } else if (startDate) {
      times.splice(0, 1);
    }
    return times;
  }

  public generateTimeOptions(): TimeOption[] {
    const validTimeStrings = this.generateTimeStrings();
    const options: TimeOption[] = validTimeStrings.map(
      (timeString: string) => ({
        disabled: false,
        value: timeString,
      }),
    );
    return options;
  }

  public validateTime = (time: string): boolean => {
    const timeRegEx = /^([0-9]|0[0-9]|1[0-9]|2[0-3]):[0-5][0-9]\s*(am|pm)?$/i;
    return Boolean(time.match(timeRegEx));
  };

  public setValue(value: string, displayValue?: string, close?: boolean): void {
    // See if the input matches any of our options, to set appropriate values
    // where they differ from displayed values.
    const timeOption = this.props.customTimeOptions
      ? this.props.customTimeOptions.find(
          (option: TimeOption) =>
            option.displayValue === (displayValue || value),
        )
      : null;
    value = timeOption?.value ? timeOption.value : value;
    displayValue = timeOption?.displayValue
      ? timeOption.displayValue
      : displayValue || value;
    this.setState({
      displayValue,
      isOpen: this.state.isOpen && close ? false : this.state.isOpen,
      time: this.getTime(value),
      value,
    });
    this.notifyChange(value);
  }

  private getTime(value?: string) {
    try {
      return parseTimeStringToTime(value || this.state.value);
    } catch (e) {
      return null;
    }
  }

  private onFocus = () => {
    this.setState({
      isOpen: true,
    });
  };

  private onBlur = (e: FormEvent<HTMLInputElement>) => {
    const displayValue = this.formatTime(e.currentTarget.value);
    this.setValue(displayValue, displayValue);
  };

  // On click, the Balloon will close and the input will blur, triggering
  // onBlur, which is why we update the input value rather than the state.
  private onClickTime = (e: MouseEvent<HTMLLIElement>) => {
    const value = e.currentTarget.getAttribute('data-value') as string;
    const displayValue =
      (e.currentTarget.getAttribute('data-display-value') as string) || value;
    this.setValue(displayValue, displayValue, true);
  };

  private setSelectedTimeRef = (e: HTMLDivElement) => (this.selectedTime = e);

  private notifyChange(value: string) {
    if (this.props.onChange) {
      try {
        const newTime = parseTimeStringToTime(value);
        this.props.onChange(newTime, value);
      } catch (e) {
        this.props.onChange(null, value);
      }
    }
  }

  private onInputChange = (e: FormEvent<HTMLInputElement>) => {
    const displayValue = e.currentTarget.value;
    if (displayValue === this.state.displayValue) {
      return;
    }
    this.setValue(displayValue);
  };

  private onKeyDown = (e: KeyboardEvent<HTMLElement>) => {
    const key = e.charCode || e.keyCode;
    if (key === 9) {
      const displayValue = this.formatTime(this.state.displayValue);
      this.setValue(displayValue, displayValue, true);
    }
  };

  private closeDropdown = () => {
    this.setState({
      isOpen: false,
    });
  };

  private timeToTimeString(time: Time) {
    return formatTime(time, DateFormats.timeString);
  }

  private formatTime(timeValue: string) {
    let formattedTimeString = this.formatShortHand(timeValue);
    if (this.validateTime?.(formattedTimeString)) {
      const tempDate = parseTimeStringToTime(formattedTimeString);
      formattedTimeString = this.timeToTimeString(tempDate);
    }
    return formattedTimeString;
  }

  private formatShortHand = (time: string): string => {
    const shortTimeRegEx = /^([0-9]|0[0-9]|1[0-9]|2[0-3])(am|pm)$/i;
    if (time.match(shortTimeRegEx)) {
      const meridiem = time.substr(time.length - 2);
      return time.replace(meridiem, `:00${meridiem}`);
    }
    return time;
  };
}
