import {
  createMountWrapperFactory,
  createShallowWrapperFactory,
} from 'tachyon-test-utils';
import { AttachedBalloon, Input } from 'twitch-core-ui';
import type { Time } from '../models';
import { TimePicker } from '.';

function getDefaultTime(): Time {
  return { hours: 3, minutes: 0 };
}

describe(TimePicker, () => {
  const propGenerator = () => {
    return {
      initialTime: getDefaultTime(),
      onChange: jest.fn(),
    };
  };

  const setupShallow = createShallowWrapperFactory(TimePicker, () =>
    propGenerator(),
  );
  const setupMount = createMountWrapperFactory(TimePicker, () =>
    propGenerator(),
  );
  describe('with user interaction on the picker', () => {
    it('opens the dropdown on focus', () => {
      const { wrapper } = setupMount();
      const input = wrapper.find('input');
      expect(input).toHaveLength(1);
      input.simulate('focus');
      wrapper.update();
      expect(wrapper.state('isOpen')).toEqual(true);
      expect(wrapper.find(AttachedBalloon)).toHaveProp('show', true);
    });

    it('closes the dropdown on tab', () => {
      const { wrapper } = setupMount();
      const input = wrapper.find(Input);
      input.simulate('focus');
      input.simulate('change', { currentTarget: { value: '11pm' } });
      input.simulate('keyDown', { keyCode: 9 });
      wrapper.update();
      expect(wrapper.state('isOpen')).toEqual(false);
      expect(wrapper.find(AttachedBalloon).props().show).toBe(false);
    });
  });

  describe('with user input time', () => {
    it('auto formats the time on blur', () => {
      const onChange = jest.fn();
      const { wrapper } = setupShallow({ onChange });
      let input = wrapper.find(Input);
      input.simulate('change', { currentTarget: { value: '16:20' } });
      input.simulate('blur', { currentTarget: { value: '16:20' } });
      expect(onChange).toHaveBeenCalledTimes(2);
      const calledDate = onChange.mock.calls[0][0] as Time;
      expect(calledDate.hours).toBe(16);
      expect(calledDate.minutes).toBe(20);
      input = wrapper.find(Input);
      expect(input).toHaveProp('value', '4:20pm');
    });

    it('auto formats shorthand time on tab', () => {
      const onChange = jest.fn();
      const { wrapper } = setupShallow({ onChange });
      let input = wrapper.find(Input);
      input.simulate('change', { currentTarget: { value: '11pm' } });
      input.simulate('keyDown', { keyCode: 9 });
      expect(onChange).toHaveBeenCalledTimes(2);
      const calledDate = onChange.mock.calls[1][0] as Time;
      expect(calledDate.hours).toBe(23);
      expect(calledDate.minutes).toBe(0);
      input = wrapper.find(Input);
      expect(input).toHaveProp('value', '11:00pm');
    });
  });

  describe('with a default time', () => {
    it('allows a time to be picked', () => {
      const initialTime = { hours: 12, minutes: 0 };
      const onChange = jest.fn();
      const { wrapper } = setupMount({ initialTime, onChange });
      const input = wrapper.find(Input);
      expect(input).toHaveLength(1);
      input.simulate('click');
      const timeLine = wrapper.find({ 'data-value': '11:30pm' }).hostNodes();
      expect(timeLine).toHaveLength(1);
      timeLine.simulate('click');
      expect(onChange.mock.calls).toHaveLength(1);
      const calledDate = onChange.mock.calls[0][0] as Time;
      expect(calledDate.hours).toBe(23);
      expect(calledDate.minutes).toBe(30);
    });
  });
});
