import { format } from 'date-fns';
import type { Time } from '../../../models';

export const DateFormats = {
  Comscore: 'YYYY-MM-DD',
  RFC3399: 'YYYY-MM-DDTHH:mm:ssZ',
  timeString: 'h:mma',
};

export enum DayValues {
  Sunday = 0,
  Monday = 1,
  Tuesday = 2,
  Wednesday = 3,
  Thursday = 4,
  Friday = 5,
  Saturday = 6,
}

export enum DateAdjustmentType {
  Prior,
  Ahead,
}

export function formatTime(time: Time, dateFormat: string): string {
  const date = new Date();
  date.setHours(time.hours);
  date.setMinutes(time.minutes);
  date.setSeconds(0);
  return format(date, dateFormat);
}

export function parseTimeStringToDate(timeStr: string, date?: Date): Date {
  const time = parseTimeStringToTime(timeStr);
  const ret = date || new Date();
  ret.setHours(time.hours, time.minutes);
  return ret;
}

export function parseTimeStringToTime(timeStr: string): Time {
  const parts = timeStr.split(':');
  if (parts.length !== 2) {
    throw new Error(`invalid time string: ${timeStr}`);
  }
  let hours = parseInt(parts[0], 10);
  const minutes = parseInt(parts[1].substring(0, 2), 10);
  const meridium = parts[1].substring(parts[1].length - 2);
  if (hours === 12 && meridium === 'am') {
    hours = 0;
  }
  if (meridium === 'pm' && hours < 12) {
    hours += 12;
  }
  if (
    isNaN(hours) ||
    hours < 0 ||
    hours > 23 ||
    isNaN(minutes) ||
    minutes < 0 ||
    minutes > 59
  ) {
    throw new Error(`invalid time string: ${timeStr}`);
  }
  return {
    hours,
    minutes,
  };
}

export function isValid(time: Time): boolean {
  if (
    time.hours > -1 &&
    time.hours < 24 &&
    time.minutes > -1 &&
    time.minutes < 60 &&
    (!time.seconds || (time.seconds > -1 && time.seconds < 60))
  ) {
    return true;
  }

  return false;
}

export function getAdjustedDateFromCurrent(
  numDays: number,
  adjustmentType: DateAdjustmentType,
): Date {
  const currDate = new Date();
  const adjustedDate =
    adjustmentType === DateAdjustmentType.Prior
      ? currDate.getDate() - numDays
      : currDate.getDate() + numDays;
  return new Date(currDate.setDate(adjustedDate));
}
