import { ClientConfiguration } from './client-configuration';
import type { TMIClientOptions, TMIIdentityOptions } from './models';
import { mockTMILogger } from './tests';
import { Utils } from './utils';

let mathSpy: jest.SpyInstance;
let utilSpy: jest.SpyInstance;

/** Bare minimum client options, used to test default values */
const bareClientOptions: TMIClientOptions = {
  connection: {
    port: 443,
    secure: true,
    server: 'irc-ws.chat.twitch.tv',
  },
  logger: mockTMILogger(),
};

/** Fully specified client options */
const fullClientOptions: TMIClientOptions = {
  connection: {
    channel: 'foobar',
    connectTimeout: 15000,
    maxReconnectAttempts: 10,
    maxReconnectInterval: 500,
    port: 443,
    reconnect: true,
    reconnectBackoff: 5,
    reconnectInterval: 500,
    secure: true,
    server: 'irc-ws.chat.twitch.tv',
  },
  identity: {
    authToken: '1234',
    username: 'profmulo',
  },
  logger: mockTMILogger(),
};

describe('ClientConfiguration', () => {
  beforeAll(() => {
    mathSpy = jest.spyOn(Math, 'random');
    mathSpy.mockReturnValue(1);

    utilSpy = jest.spyOn(Utils, 'randomGuestUsername');
    utilSpy.mockReturnValue('justinfan01');
  });

  afterAll(() => {
    mathSpy.mockRestore();
    utilSpy.mockRestore();
  });

  it('gets constructed with correct default values', () => {
    const clientConfiguration = new ClientConfiguration(bareClientOptions);

    expect(clientConfiguration).toBeTruthy();
    expect(clientConfiguration).toMatchObject({
      authToken: undefined,
      baseReconnectInterval: 1000,
      channel: undefined,
      connectTimeout: 10000,
      connectUrl: `wss://${fullClientOptions.connection.server}:${fullClientOptions.connection.port}`,
      maxReconnectAttempts: Infinity,
      maxReconnectInterval: 30000,
      password: 'SCHMOOPIIE',
      pingInterval: 310000,
      pingTimeout: 10000,
      port: bareClientOptions.connection.port,
      reconnectBackoff: 2000,
      reconnectJitter: 5000,
      secure: bareClientOptions.connection.secure,
      server: bareClientOptions.connection.server,
      username: 'justinfan01',
    });
  });

  it('gets constructed with provided values', () => {
    const clientConfiguration = new ClientConfiguration(fullClientOptions);

    expect(clientConfiguration).toBeTruthy();
    expect(clientConfiguration).toMatchObject({
      authToken: fullClientOptions.identity!.authToken,
      baseReconnectInterval: fullClientOptions.connection.reconnectInterval,
      channel: `#${fullClientOptions.connection.channel}`,
      connectTimeout: fullClientOptions.connection.connectTimeout,
      connectUrl: `wss://${fullClientOptions.connection.server}:${fullClientOptions.connection.port}`,
      maxReconnectAttempts: fullClientOptions.connection.maxReconnectAttempts,
      maxReconnectInterval: fullClientOptions.connection.maxReconnectInterval,
      password: `oauth:${fullClientOptions.identity!.authToken}`,
      pingInterval: 310000,
      pingTimeout: 10000,
      port: fullClientOptions.connection.port,
      reconnectBackoff: fullClientOptions.connection.reconnectBackoff,
      reconnectJitter: 5000,
      secure: fullClientOptions.connection.secure,
      server: fullClientOptions.connection.server,
      username: fullClientOptions.identity!.username,
    });
  });

  describe('updateIdentity()', () => {
    it('correctly updates identity properties', () => {
      const mockIdentity: TMIIdentityOptions = {
        authToken: '5678',
        username: 'kappa',
      };

      const clientConfiguration = new ClientConfiguration(fullClientOptions);
      clientConfiguration.updateIdentity(mockIdentity);

      expect(clientConfiguration).toMatchObject({
        authToken: mockIdentity.authToken,
        connectUrl: `wss://${fullClientOptions.connection.server}:${fullClientOptions.connection.port}`,
        password: `oauth:${mockIdentity.authToken}`,
        username: mockIdentity.username,
      });
    });
  });

  describe('getReconnectDelay()', () => {
    it('returns the correct delay for incremental attempts', () => {
      const clientConfiguration = new ClientConfiguration(bareClientOptions);
      // 1000 base + (attempt * 2000 reconnectBackoff) + 5000 jitter
      expect(clientConfiguration.getReconnectDelay(1)).toBe(8000);
      expect(clientConfiguration.getReconnectDelay(2)).toBe(10000);
      expect(clientConfiguration.getReconnectDelay(3)).toBe(12000);
    });
  });
});
