import type { TMIClientOptions, TMIIdentityOptions } from './models';
import { Utils } from './utils';

export class ClientConfiguration {
  public server: string;
  public readonly port: number;
  public readonly secure: boolean;
  public readonly maxReconnectAttempts: number;
  // The longest to wait between reconnects
  public readonly maxReconnectInterval: number;
  public readonly reconnectBackoff: number;
  public readonly baseReconnectInterval: number;
  public readonly connectTimeout: number;
  public readonly channel: string | undefined;

  public readonly reconnectJitter: number;
  public readonly pingInterval: number;
  // 10 seconds;
  public readonly pingTimeout: number = 10000;
  public skipAutoRejoin = false;

  public connectUrl = '';
  public username = '';
  public authToken: string | undefined = '';
  public password = '';

  // 5 minutes
  private readonly basePingInterval = 300000;

  // 10 seconds
  private readonly maxPingJitter = 10000;

  // 5 seconds
  private readonly maxReconnectJitter = 5000;

  constructor(opts: TMIClientOptions) {
    if (!opts) {
      throw new Error('Options must be supplied.');
    }
    if (!opts.connection) {
      throw new Error('Connection options must be supplied.');
    }

    this.channel =
      opts.connection.channel && Utils.channel(opts.connection.channel);

    // Reconnect Config
    this.maxReconnectAttempts =
      opts.connection.maxReconnectAttempts || Infinity;
    this.maxReconnectInterval = opts.connection.maxReconnectInterval || 30000;
    this.reconnectBackoff = opts.connection.reconnectBackoff || 2000;
    this.baseReconnectInterval = opts.connection.reconnectInterval || 1000;

    // Connection Config
    this.port = opts.connection.port;
    this.secure = opts.connection.secure;
    this.server = opts.connection.server;
    this.connectTimeout = opts.connection.connectTimeout || 10000;
    this.pingInterval =
      this.basePingInterval + Math.round(this.maxPingJitter * Math.random());
    this.reconnectJitter = Math.round(this.maxReconnectJitter * Math.random());
    this.updateIdentity(opts.identity);
  }

  public updateIdentity(opts?: TMIIdentityOptions): void {
    this.authToken = opts?.authToken;
    this.password = opts?.authToken ? `oauth:${opts.authToken}` : 'SCHMOOPIIE';
    this.username = opts?.username ?? Utils.randomGuestUsername();
    this.setConnectUrl();
  }

  public updateServerHost(host: string): void {
    this.server = host;
    this.setConnectUrl();
  }

  public updateRejoinConfiguration(skipAutoRejoin: boolean): void {
    this.skipAutoRejoin = skipAutoRejoin;
  }

  public getReconnectDelay(attempts: number): number {
    const currentBackoff = this.reconnectBackoff * attempts;
    const interval = Math.min(
      this.baseReconnectInterval + currentBackoff,
      this.maxReconnectInterval,
    );
    return interval + this.reconnectJitter;
  }

  private setConnectUrl(): void {
    this.connectUrl = `${this.secure ? 'wss://' : 'ws://'}${this.server}:${
      this.port
    }`;
  }
}
