import { TMICommand } from './tmi-command';
import type { TMICommandRequest, TMICommandResponse } from './tmi-command-args';

export interface ColorRequest extends TMICommandRequest {
  color: string;
}

export interface ColorResponse extends TMICommandResponse {
  succeeded: boolean;
}

export class ColorCommand extends TMICommand<ColorRequest, ColorResponse> {
  public getCommandText(data: ColorRequest): Array<string> {
    return [`/color ${data.color}`];
  }

  protected override async beforeSendCommand(
    data: ColorRequest,
  ): Promise<void> {
    const userState = this.session.getUserState(data.channel);
    if (!userState) {
      this.logger.warn(
        'Unable to optimistically update color. No user state for channel.',
        { data },
      );
      return;
    }

    if (!this.isHexColor(data.color)) {
      this.logger.warn(
        'Unable to optimistically update color. Supplied color must be in the hex format.',
        { data },
      );
      return;
    }

    userState.color = data.color;
    this.logger.debug('Optimistically updated chat color.', {
      color: data.color,
    });
  }

  private isHexColor(color: string) {
    return /^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/.test(color);
  }
}
