import type { EventProcessors } from '../event-processors';
import {
  mockEventProcessors,
  mockTMIConnection,
  mockTMISession,
} from '../tests';
import type { TMIConnection } from '../tmi-connection';
import { SendMessageCommand } from './send-message';

const VALID_ACTIONS = [
  '/me ',
  '/me jokes',
  '.me ',
  '.me says a joke',
  '/ME ',
  '/ME says hi',
];

const INVALID_ACTIONS = [
  '/me',
  '.me',
  '\\me',
  '\\me ',
  '\\me makes a joke',
  '/meta joke',
  '.meta joke about a joke',
  '\\meta hahaha',
  'me is not funny',
  ' /me does not like to laugh',
  ' ',
  ' /me spoils the joke',
  ' /ME cries',
  '\\ME fixes a bug',
];

function setup() {
  const tmiConnection: TMIConnection = mockTMIConnection();
  const tmiSession = mockTMISession();
  const eventProcessors: EventProcessors = mockEventProcessors();
  const command = new SendMessageCommand(
    tmiConnection,
    tmiSession,
    eventProcessors,
  );

  tmiSession.getUserState = jest.fn().mockReturnValue({});

  return {
    command,
    eventProcessors,
    tmiConnection,
    tmiSession,
  };
}
describe('send-message', () => {
  it('can be instantiated', () => {
    const { command } = setup();
    expect(command).toBeDefined();
  });

  describe('.getCommandText()', () => {
    it('extracts the message', () => {
      const { command } = setup();
      const data = { message: 'testmessage' };
      expect(command.getCommandText(data)).toEqual([data.message]);
    });
  });

  describe('.execute()', () => {
    it('doesnt do anything if it cant find a valid channel', () => {
      const { command, eventProcessors, tmiSession } = setup();
      tmiSession.getUserState = jest.fn().mockReturnValue(null);
      command.execute({ message: 'this should not do anything' }, 1000);
      expect(eventProcessors.action).not.toHaveBeenCalled();
      expect(eventProcessors.chat).not.toHaveBeenCalled();
    });

    describe('/me', () => {
      for (const message of VALID_ACTIONS) {
        it(`correctly recognizes '${message}' as an action`, () => {
          const { command, eventProcessors } = setup();
          command.execute({ channel: 'testchannel', message }, 1000);
          expect(eventProcessors.action).toHaveBeenCalled();
          expect(eventProcessors.chat).not.toHaveBeenCalled();
        });
      }

      for (const message of INVALID_ACTIONS) {
        it(`correctly rejects '${message}' as an action`, () => {
          const { command, eventProcessors } = setup();
          command.execute({ channel: 'testchannel', message }, 1000);
          expect(eventProcessors.action).not.toHaveBeenCalled();
        });
      }
    });

    it('invokes eventProcessor.chat() for regular messages', () => {
      const { command, eventProcessors } = setup();
      command.execute(
        { channel: 'testchannel', message: 'this is a regular message' },
        1000,
      );
      expect(eventProcessors.action).not.toHaveBeenCalled();
      expect(eventProcessors.chat).toHaveBeenCalled();
    });

    it('discards non-action commands', () => {
      const { command, eventProcessors } = setup();
      command.execute(
        { channel: 'testchannel', message: '/othercommand' },
        1000,
      );
      expect(eventProcessors.action).not.toHaveBeenCalled();
      expect(eventProcessors.chat).not.toHaveBeenCalled();
    });

    it('parses emotes', () => {
      const { command, eventProcessors } = setup();
      command.execute({ channel: 'testchannel', message: 'hello Kappa' }, 1000);
      expect(eventProcessors.chat).toHaveBeenCalledWith(
        expect.objectContaining({
          message: expect.objectContaining({
            user: expect.objectContaining({
              emotes: expect.objectContaining({
                6: { id: expect.any(String), startIndex: 6 },
              }),
            }),
          }),
        }),
      );
    });
  });
});
