import type { EventProcessors } from '../event-processors';
import type {
  TMIChatMessage,
  TMIEmotePositionMap,
  TMISession,
  TMIUser,
} from '../models';
import { TMIChatEventType } from '../models';
import type { TMIAdditionalMetadata } from '../tmi-command-processor';
import type { TMIConnection } from '../tmi-connection';
import { generateEmotePositions } from '../utils';
import { TMICommand } from './tmi-command';
import type { TMICommandRequest, TMICommandResponse } from './tmi-command-args';

export interface SendMessageRequest extends TMICommandRequest {
  additionalMetadata?: TMIAdditionalMetadata;
  clientNonce?: string;
  message: string;
}

export class SendMessageCommand extends TMICommand<
  SendMessageRequest,
  TMICommandResponse
> {
  protected readonly events: EventProcessors;

  constructor(
    connection: TMIConnection,
    session: TMISession,
    events: EventProcessors,
  ) {
    super(connection, session);

    this.events = events;
  }

  public getCommandText(data: SendMessageRequest): Array<string> {
    return [data.message];
  }

  protected override async beforeSendCommand(
    data: SendMessageRequest,
  ): Promise<void> {
    this.logger.debug('beforeSendCommand', { data });

    if (!data.channel) {
      this.logger.warn('Unable to send message. No channel specified.');
      return;
    }

    const isCommand = this.isCommand(data.message);
    const isAction = this.isAction(data.message);

    if (isCommand && !isAction) {
      return;
    }

    const messageContent = isAction ? data.message.substr(4) : data.message;

    // Parse emotes
    const emotes: TMIEmotePositionMap = generateEmotePositions(
      messageContent,
      this.session.emoteMap,
    );

    const userState = this.session.getUserState(data.channel);
    if (!userState) {
      this.logger.warn('Unable to send message. No channel user state.', {
        data,
      });
      return;
    }

    const userstate: TMIUser = {
      ...userState,
      emotes,
      id: Date.now().toString(),
    };

    const chatMessage: TMIChatMessage = {
      body: messageContent,
      id: '',
      timestamp: Date.now(),
      user: userstate,
    };

    const chatData = {
      additionalMetadata: data.additionalMetadata,
      channel: data.channel,
      message: chatMessage,
      sentByCurrentUser: true,
      timestamp: Date.now(),
    };

    if (isAction) {
      /**
       * Unlike action messages that come thru IRC, locally rendered action messages
       * will have the exact same string in the action field and the message.body field
       * This is because the locally rendered messages are not actually sent to the backend
       * and thus do not need the \u0001ACTION command headers
       */
      this.events.action({
        ...chatData,
        action: messageContent,
        type: TMIChatEventType.Action,
      });
    } else {
      this.events.chat({
        ...chatData,
        type: TMIChatEventType.Message,
      });
    }
  }

  private isAction(message: string) {
    return (
      this.isCommand(message) && message.substr(1, 3).toLowerCase() === 'me '
    );
  }

  private isCommand(message: string) {
    return (
      (message.startsWith('.') && !message.startsWith('..')) ||
      message.startsWith('/')
    );
  }
}
