import { mockTMIConnection, mockTMISession } from '../tests';
import { TMICommand } from './tmi-command';
import type { TMICommandRequest, TMICommandResponse } from './tmi-command-args';

class MockTMICommand extends TMICommand<TMICommandRequest, TMICommandResponse> {
  public getCommandText() {
    return ['test'];
  }

  public override beforeSendCommand() {
    return Promise.resolve();
  }
}

const setup = () => {
  const connection = mockTMIConnection();
  const command = new MockTMICommand(connection, mockTMISession());

  return {
    command,
    connection,
  };
};

describe('TMICommand', () => {
  it('can be constructed', () => {
    const { command } = setup();

    expect(command).toBeTruthy();
    expect(command).toBeInstanceOf(TMICommand);
  });

  describe('executing a command', () => {
    afterEach(() => {
      // don't let timeouts that aren't relevant to tests trigger unhandled rejections
      jest.clearAllTimers();
    });

    it('gets the command timeout from the connection only if not specified', () => {
      const { command, connection } = setup();
      jest.spyOn(connection, 'getCommandTimeout');

      // Timeout is provided, should not be called
      command.execute({ channel: 'testchannel' }, 1000);
      expect(connection.getCommandTimeout).not.toHaveBeenCalled();

      // Timeout is omitted, should be called
      command.execute({ channel: 'testchannel' });
      expect(connection.getCommandTimeout).toHaveBeenCalledTimes(1);
    });

    it('calls the getCommandText and beforeSendCommand methods of the inheriting command', async () => {
      const { command } = setup();
      jest.spyOn(command, 'getCommandText');
      jest.spyOn(command, 'beforeSendCommand');

      command.execute({ channel: 'testchannel' });

      expect(command.getCommandText).toHaveBeenCalledTimes(1);
      expect(command.beforeSendCommand).toHaveBeenCalledTimes(1);
    });

    it('returns a promise that rejects if there is no response', async () => {
      const { command } = setup();

      const result = command.execute({ channel: 'testchannel' }, 1000);
      await Promise.resolve();

      // Simulate the timeout occuring
      jest.runOnlyPendingTimers();

      expect(result).toBeInstanceOf(Promise);
      await expect(result).rejects.toEqual({
        reason: expect.stringContaining('No response'),
      });
    });

    it('returns a promise that rejects when the response is received after the timeout', async () => {
      const { command } = setup();

      // Execute the command
      const result = command.execute({ channel: 'testchannel' }, 1000);
      await Promise.resolve();

      // Simulate the timeout occuring
      jest.runOnlyPendingTimers();

      // The response is signaled after the timeout
      command.signal({});

      // Confirm that the promise still rejects
      await expect(result).rejects.toEqual({
        reason: expect.stringContaining('No response'),
      });
    });

    it('returns a promise that resolves if a response is signaled', async () => {
      const { command } = setup();

      const result = command.execute({ channel: 'testchannel' }, 1000);
      await Promise.resolve();

      command.signal({});

      expect(result).toBeInstanceOf(Promise);
      await expect(result).resolves.toEqual({
        request: { channel: '#testchannel' },
        response: { duration: 0 },
      });
    });

    it('sends the formatted command text through the TMIConnection', async () => {
      const { command, connection } = setup();

      command.execute({ channel: 'testchannel' });
      await Promise.resolve();

      expect(connection.send).toHaveBeenLastCalledWith(
        'PRIVMSG #testchannel :test',
      );
    });

    it('sends the formatted command with reply tags through the TMIConnection', async () => {
      const { command, connection } = setup();
      command.execute({
        additionalMetadata: {
          reply: {
            parentDeleted: false,
            parentDisplayName: 'this is a displayName',
            parentMessageBody: 'test message body',
            parentMsgId: 'ajkwe-jkdf-ajde',
            parentUid: '134242442',
            parentUserLogin: 'nerds235',
          },
        },
        channel: 'testchannel',
      });
      await Promise.resolve();

      expect(connection.send).toHaveBeenLastCalledWith(
        '@reply-parent-msg-id=ajkwe-jkdf-ajde PRIVMSG #testchannel :test',
      );
    });

    it('sends the formatted command with a crowd chant tag through the TMIConnection', async () => {
      const { command, connection } = setup();
      command.execute({
        additionalMetadata: {
          crowdChantParentMessageID: 'abcd-efgh-ijkl',
        },
        channel: 'testchannel',
      });
      await Promise.resolve();

      expect(connection.send).toHaveBeenLastCalledWith(
        '@crowd-chant-parent-msg-id=abcd-efgh-ijkl PRIVMSG #testchannel :test',
      );
    });
  });
});
