import type { TMILogger, TMISession } from '../models';
import type { TMIConnection } from '../tmi-connection';
import { Utils } from '../utils';
import type { TMICommandRequest, TMICommandResponse } from './tmi-command-args';

export interface TMICommandQueue<TRequest, TResponse> {
  request: TRequest;
  resolve: (result: TMICommandResult<TRequest, TResponse>) => void;
  startTimestamp: number;
  timeoutHandle: number;
}

export interface TMICommandResult<TRequest, TResponse> {
  request: TRequest;
  response: TResponse;
}

export abstract class TMICommand<
  TRequest extends TMICommandRequest,
  TResponse extends TMICommandResponse,
> {
  protected readonly connection: TMIConnection;
  protected readonly session: TMISession;

  protected readonly logger: TMILogger;
  private pendingCommands: Array<TMICommandQueue<TRequest, TResponse>> = [];

  constructor(connection: TMIConnection, session: TMISession) {
    this.connection = connection;
    this.session = session;
    this.logger = connection.logger;
  }

  public async execute(
    data: TRequest,
    timeout?: number,
  ): Promise<TMICommandResult<TRequest, TResponse>> {
    if (data.channel) {
      data.channel = Utils.channel(data.channel);
    }
    return this.sendCommand(data, timeout);
  }

  public signal = (data: TResponse): void => {
    const now = Date.now();
    const { pendingCommands } = this;
    if (pendingCommands) {
      this.logger.debug('Command received completion signal', {
        commands: pendingCommands.slice(0),
        data,
      });

      const pendingCallback = pendingCommands.shift();
      if (pendingCallback) {
        data.duration = now - pendingCallback.startTimestamp;
        window.clearTimeout(pendingCallback.timeoutHandle);
        pendingCallback.resolve({
          request: pendingCallback.request,
          response: data,
        });
      }
    } else {
      this.logger.debug(
        'Command received completion signal, but no pending commands registered',
        { data },
      );
    }
  };

  protected beforeSendCommand(_data: TRequest): Promise<void> {
    return Promise.resolve();
  }

  protected sendCommand(
    data: TRequest,
    timeout?: number,
  ): Promise<TMICommandResult<TRequest, TResponse>> {
    const commands = this.getCommandText(data);
    const commandTimeout = timeout || this.connection.getCommandTimeout();
    const startTimestamp = Date.now();

    this.logger.debug('Creating command promise', {
      commandText: commands.join(';'),
      timeout,
    });

    return new Promise((resolve, reject) => {
      let timeoutHandle = 0;

      // Set a promise timeout
      timeoutHandle = window.setTimeout(() => {
        this.logger.debug('Command timed out', { commands, data, timeout });
        reject({ reason: `No response from Twitch after ${commandTimeout}` });
      }, commandTimeout);

      // Run any optional code before executing commands
      this.beforeSendCommand(data).then(() => {
        this.registerCommand(timeoutHandle, data, startTimestamp, resolve);

        for (const commandText of commands) {
          let formattedCommandText = data.channel
            ? `PRIVMSG ${Utils.channel(data.channel)} :${commandText}`
            : commandText;
          const tagString = this.formatTags(data);

          if (tagString !== '') {
            formattedCommandText = `${tagString} ${formattedCommandText}`;
          }
          this.logger.debug('Executing command', { formattedCommandText });
          this.connection.send(formattedCommandText);
        }
      });
    });
  }

  // This function will parse the tags and return it in a string like '@client-nonce=abc-123-ejf`. Use that string to prepend the IRC message.
  private formatTags(data: TRequest): string {
    if (!data.additionalMetadata) {
      return '';
    }
    let textWithTags = '';

    if (data.additionalMetadata.clientNonce) {
      textWithTags += `client-nonce=${data.additionalMetadata.clientNonce};`;
    }

    if (data.additionalMetadata.crowdChantParentMessageID) {
      textWithTags += `crowd-chant-parent-msg-id=${data.additionalMetadata.crowdChantParentMessageID};`;
    }

    if (data.additionalMetadata.reply) {
      textWithTags += `reply-parent-msg-id=${data.additionalMetadata.reply.parentMsgId};`;
    }

    if (textWithTags !== '') {
      textWithTags = `@${textWithTags}`;
      if (textWithTags[textWithTags.length - 1] === ';') {
        // Remove the last semicolon
        textWithTags = textWithTags.substring(0, textWithTags.length - 1);
      }
    }

    return textWithTags;
  }

  private registerCommand(
    timeoutHandle: number,
    request: TRequest,
    startTimestamp: number,
    resolve: (data: TMICommandResult<TRequest, TResponse>) => void,
  ) {
    if (!this.pendingCommands) {
      this.pendingCommands = [];
    }

    const pendingCommand = { request, resolve, startTimestamp, timeoutHandle };
    this.pendingCommands.push(pendingCommand);
  }

  protected abstract getCommandText(data: TRequest): string[];
}
