import { MessageProcessor } from './message-processor';
import { TMIChannel, TMIChatEventType, TMIFlagCategory } from './models';
import type { CreateMessageOptions } from './tests';
import {
  createGoalTags,
  createMessage,
  createPrefixWithUsername,
  createSubscriptionFunFactFollowingAgeTags,
  createSubscriptionFunFactMostUsedEmoteTags,
  createUserTags,
  expectNoop,
  expectNoticeEventRaised,
  mockAnonymousUser,
  mockEventProcessors,
  mockGoalData,
  mockLoggedInUser,
  mockSubscriptionFunFactFollowingAge,
  mockSubscriptionFunFactMostUsedEmote,
  mockTMIConnection,
  mockTMILogger,
  mockTMISession,
} from './tests';
import { TMIParser } from './tmi-parser';

describe('MessageProcessor', () => {
  const defaults: CreateMessageOptions = {};

  const setup = (options: CreateMessageOptions = {}) => {
    const { message, messageParts } = createMessage(options);
    const connection = mockTMIConnection();
    const session = mockTMISession();
    const events = mockEventProcessors();

    const processor = new MessageProcessor(
      new TMIParser(mockTMILogger()),
      events as any,
      session,
      connection as any,
    );

    return {
      channel: messageParts.channel,
      command: messageParts.command,
      connection,
      events,
      // MessageProcessor specifically uses the connection's logger as its own
      logger: connection.logger,
      message,
      processor,
      session,
    };
  };

  it('can be constructed', () => {
    const { processor } = setup();
    expect(processor).toBeTruthy();
  });

  describe('processMessage()', () => {
    describe('messages without a prefix', () => {
      it('handles a PING command', () => {
        const { connection, message, processor } = setup({ command: 'PING' });
        processor.processMessage(message);

        expect(connection.pong).toHaveBeenCalledTimes(1);
      });

      it('handles a PONG command', () => {
        const { connection, message, processor } = setup({ command: 'PONG' });
        processor.processMessage(message);

        expect(connection.commands.ping.signal).toHaveBeenLastCalledWith({
          channel: undefined,
          msgid: '',
        });
      });

      it('handles other commands as no-ops', () => {
        const { connection, events, message, processor } = setup({
          command: 'FOO',
        });
        processor.processMessage(message);

        expectNoop(events, connection);
      });
    });

    describe('messages with prefix "tmi.twitch.tv"', () => {
      beforeEach(() => {
        defaults.prefix = 'tmi.twitch.tv';
      });

      it('handles a 001 command to set the session username', () => {
        const { message, processor, session } = setup({
          ...defaults,
          channel: null,
          command: '001',
          params: ['foo'],
        });

        processor.processMessage(message);
        expect(session.username).toBe('foo');
      });

      it('handles a 372 command to signal connection to the server', () => {
        const { channel, connection, message, processor } = setup({
          ...defaults,
          command: '372',
          msgid: 'go',
        });

        processor.processMessage(message);
        expect(connection.commands.connect.signal).toHaveBeenLastCalledWith({
          channel,
          msgid: 'go',
          succeeded: true,
        });
      });

      describe('handling a NOTICE command', () => {
        defaults.command = 'NOTICE';

        it('handles a "subs_on" message type', () => {
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid: 'subs_on',
          });

          processor.processMessage(message);

          // subscriberModeOn command is signaled
          expect(
            connection.commands.subscriberModeOn.signal,
          ).toHaveBeenLastCalledWith({
            channel,
            msgid: 'subs_on',
            succeeded: true,
          });

          // subscribers event is raised
          expect(events.subscribers).toHaveBeenLastCalledWith({
            channel,
            enabled: true,
          });
        });

        it('handles a "subs_off" message type', () => {
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid: 'subs_off',
          });

          processor.processMessage(message);

          // subscriberModeOff command is signaled
          expect(
            connection.commands.subscriberModeOff.signal,
          ).toHaveBeenLastCalledWith({
            channel,
            msgid: 'subs_off',
            succeeded: true,
          });

          // subscribers event is raised
          expect(events.subscribers).toHaveBeenLastCalledWith({
            channel,
            enabled: false,
          });
        });

        it('handles an "emote_only_on" message type', () => {
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid: 'emote_only_on',
          });

          processor.processMessage(message);

          // emoteOnlyModeOn command is signaled
          expect(
            connection.commands.emoteOnlyModeOn.signal,
          ).toHaveBeenLastCalledWith({
            channel,
          });

          // emoteonlymode event is raised
          expect(events.emoteonlymode).toHaveBeenLastCalledWith({
            channel,
            enabled: true,
          });
        });

        it('handles an "emote_only_off" message type', () => {
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid: 'emote_only_off',
          });

          processor.processMessage(message);

          // emoteOnlyModeOn command is signaled
          expect(
            connection.commands.emoteOnlyModeOff.signal,
          ).toHaveBeenLastCalledWith({
            channel,
          });

          // emoteonlymode event is raised
          expect(events.emoteonlymode).toHaveBeenLastCalledWith({
            channel,
            enabled: false,
          });
        });

        it('handles a "slow_on" message type', () => {
          const { channel, connection, message, processor } = setup({
            ...defaults,
            msgid: 'slow_on',
          });

          processor.processMessage(message);

          // slowModeOn command is signaled
          expect(
            connection.commands.slowModeOn.signal,
          ).toHaveBeenLastCalledWith({
            channel,
            msgid: 'slow_on',
            succeeded: true,
          });
        });

        it('handles a "slow_off" message type', () => {
          const { channel, connection, message, processor } = setup({
            ...defaults,
            msgid: 'slow_off',
          });

          processor.processMessage(message);

          // slowModeOff command is signaled
          expect(
            connection.commands.slowModeOff.signal,
          ).toHaveBeenLastCalledWith({
            channel,
            msgid: 'slow_off',
            succeeded: true,
          });
        });

        it('handles various followers_on message types', () => {
          const messageTypes = ['followers_on', 'followers_on_zero'];

          for (const msgid of messageTypes) {
            const { channel, connection, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            // followersOnlyOn command is signaled
            expect(
              connection.commands.followersOnlyOn.signal,
            ).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: true,
            });
          }
        });

        it('handles a "followers_off" message type', () => {
          const { channel, connection, message, processor } = setup({
            ...defaults,
            msgid: 'followers_off',
          });

          processor.processMessage(message);

          // followersOnlyOff command is signaled
          expect(
            connection.commands.followersOnlyOff.signal,
          ).toHaveBeenLastCalledWith({
            channel,
            msgid: 'followers_off',
            succeeded: true,
          });
        });

        it('handles a "r9k_on" message type', () => {
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid: 'r9k_on',
          });

          processor.processMessage(message);

          // r9kModeOn command is signaled
          expect(connection.commands.rk9ModeOn.signal).toHaveBeenLastCalledWith(
            {
              channel,
            },
          );

          // r9kmode event is raised
          expect(events.r9kmode).toHaveBeenLastCalledWith({
            channel,
            enabled: true,
          });
        });

        it('handles a "r9k_off" message type', () => {
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid: 'r9k_off',
          });

          processor.processMessage(message);

          // r9kModeOff command is signaled
          expect(
            connection.commands.rk9ModeOff.signal,
          ).toHaveBeenLastCalledWith({
            channel,
          });

          // r9kmode event is raised
          expect(events.r9kmode).toHaveBeenLastCalledWith({
            channel,
            enabled: false,
          });
        });

        it('handles a "room_mods" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'room_mods',
            params: [
              ':The moderators of this room are: shroud, ninja, profmulo',
            ],
          });

          processor.processMessage(message);

          // mods event is raised
          expect(events.mods).toHaveBeenLastCalledWith({
            channel,
            usernames: ['shroud', 'ninja', 'profmulo'],
          });
        });

        it('handles a "no_mods" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'no_mods',
          });

          processor.processMessage(message);

          // mods event is raised
          expect(events.mods).toHaveBeenLastCalledWith({
            channel,
            usernames: [],
          });
        });

        it('handles a "msg_channel_suspended" message type', () => {
          const msgid = 'msg_channel_suspended';
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          expectNoticeEventRaised(events, { body: '', channel, msgid });
        });

        it('handles various ban failure message types', () => {
          const messageTypes = [
            'already_banned',
            'bad_ban_admin',
            'bad_ban_broadcaster',
            'bad_ban_global_mod',
            'bad_ban_self',
            'bad_ban_staff',
            'usage_ban',
          ];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            // ban command is signaled
            expect(connection.commands.ban.signal).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });

            // notice event is raised
            expectNoticeEventRaised(events, { body: '', channel, msgid });
          }
        });

        it('handles a "ban_success" message type', () => {
          const msgid = 'ban_success';

          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          // ban command is signaled
          expect(connection.commands.ban.signal).toHaveBeenLastCalledWith({
            channel,
            msgid,
            succeeded: true,
          });

          // notice event is raised
          expectNoticeEventRaised(events, { body: '', channel, msgid });
        });

        it('handles a "usage_clear" message type', () => {
          const msgid = 'usage_clear';
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          // notice event is raised
          expectNoticeEventRaised(events, { body: '', channel, msgid });
        });

        it('handles a "usage_mods" message type', () => {
          const msgid = 'usage_mods';
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          // notice event is raised
          expectNoticeEventRaised(events, { body: '', channel, msgid });
        });

        it('handles a "mod_success" message type', () => {
          const msgid = 'mod_success';
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          // notice event is raised
          expectNoticeEventRaised(events, { body: '', channel, msgid });
        });

        it('handles various mod failure message types', () => {
          const messageTypes = ['usage_mod', 'bad_mod_banned', 'bad_mod_mod'];

          for (const msgid of messageTypes) {
            const { channel, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            // notice event is raised
            expectNoticeEventRaised(events, { body: '', channel, msgid });
          }
        });

        it('handles a "unmod_success" message type', () => {
          const msgid = 'unmod_success';
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          // notice event is raised
          expectNoticeEventRaised(events, { body: '', channel, msgid });
        });

        it('handles various unmod failure message types', () => {
          const messageTypes = ['usage_unmod', 'bad_unmod_mod'];

          for (const msgid of messageTypes) {
            const { channel, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            // notice event is raised
            expectNoticeEventRaised(events, { body: '', channel, msgid });
          }
        });

        it('handles a "color_changed" message type', () => {
          const msgid = 'color_changed';
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          expectNoticeEventRaised(events, { body: '', channel, msgid });
        });

        it('handles various color failure message types', () => {
          const messageTypes = ['usage_color', 'turbo_only_color'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            // notice event raised
            expectNoticeEventRaised(events, { body: '', channel, msgid });

            // color command signaled
            expect(connection.commands.color.signal).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles a "commercial_success" message type', () => {
          const msgid = 'commercial_success';
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          // notice event raised
          expectNoticeEventRaised(events, { body: '', channel, msgid });

          // commercial command signaled
          expect(
            connection.commands.commercial.signal,
          ).toHaveBeenLastCalledWith({
            channel,
            msgid,
            succeeded: true,
          });
        });

        it('handles various commercial failure message types', () => {
          const messageTypes = ['usage_commercial', 'bad_commercial_error'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            // notice event raised
            expectNoticeEventRaised(events, { body: '', channel, msgid });

            // commercial command signaled
            expect(
              connection.commands.commercial.signal,
            ).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles a "hosts_remaining" message type', () => {
          const msgid = 'hosts_remaining';
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
            params: ['4'],
          });

          processor.processMessage(message);

          expect(connection.commands.host.signal).toHaveBeenLastCalledWith({
            channel,
            msgid,
            remainingHost: 4,
            succeeded: true,
          });

          expectNoticeEventRaised(events, { body: '4', channel, msgid });
        });

        it('handles various host failure message types', () => {
          const messageTypes = [
            'bad_host_hosting',
            'bad_host_rate_exceeded',
            'bad_host_error',
            'usage_host',
          ];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expect(connection.commands.host.signal).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });

            expectNoticeEventRaised(events, { body: '', channel, msgid });
          }
        });

        it('handles various r9kbeta success types', () => {
          const messageTypes = ['already_r9k_on', 'usage_r9k_on'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(
              connection.commands.rk9ModeOn.signal,
            ).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles various r9kbeta failure types', () => {
          const messageTypes = ['already_r9k_off', 'usage_r9k_off'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(
              connection.commands.rk9ModeOff.signal,
            ).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles a "timeout_success"', () => {
          const msgid = 'timeout_success';
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          expectNoticeEventRaised(events, { body: '', channel, msgid });

          expect(connection.commands.timeout.signal).toHaveBeenLastCalledWith({
            channel,
            msgid,
            succeeded: true,
          });
        });

        it('handles various subscribersoff failure message types', () => {
          const messageTypes = ['already_subs_off', 'usage_subs_off'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(
              connection.commands.subscriberModeOff.signal,
            ).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles various subscribers failure message types', () => {
          const messageTypes = ['already_subs_on', 'usage_subs_on'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(
              connection.commands.subscriberModeOn.signal,
            ).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles various emoteonlyoff failure message types', () => {
          const messageTypes = [
            'already_emote_only_off',
            'usage_emote_only_off',
          ];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(
              connection.commands.emoteOnlyModeOff.signal,
            ).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles various emoteonly failure message types', () => {
          const messageTypes = ['already_emote_only_on', 'usage_emote_only_on'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(
              connection.commands.emoteOnlyModeOn.signal,
            ).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles a "usage_slow_on" message type', () => {
          const msgid = 'usage_slow_on';
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          expectNoticeEventRaised(events, { body: '', channel, msgid });

          expect(
            connection.commands.slowModeOn.signal,
          ).toHaveBeenLastCalledWith({
            channel,
            msgid,
            succeeded: false,
          });
        });

        it('handles a "usage_slow_off" message type', () => {
          const msgid = 'usage_slow_off';
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          expectNoticeEventRaised(events, { body: '', channel, msgid });

          expect(
            connection.commands.slowModeOff.signal,
          ).toHaveBeenLastCalledWith({
            channel,
            msgid,
            succeeded: false,
          });
        });

        it('handles various timeout failure message types', () => {
          const messageTypes = [
            'usage_timeout',
            'bad_timeout_admin',
            'bad_timeout_broadcaster',
            'bad_timeout_duration',
            'bad_timeout_global_mod',
            'bad_timeout_self',
            'bad_timeout_staff',
          ];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(connection.commands.timeout.signal).toHaveBeenLastCalledWith(
              {
                channel,
                msgid,
                succeeded: false,
              },
            );
          }
        });

        it('handles an "unban_success" message type', () => {
          const msgid = 'unban_success';
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          expectNoticeEventRaised(events, { body: '', channel, msgid });

          expect(connection.commands.unban.signal).toHaveBeenLastCalledWith({
            channel,
            msgid,
            succeeded: true,
          });
        });

        it('handles various unban failure message types', () => {
          const messageTypes = ['usage_unban', 'bad_unban_no_ban'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(connection.commands.unban.signal).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles various unhost failure message types', () => {
          const messageTypes = ['usage_unhost', 'not_hosting'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });
            expect(connection.commands.unhost.signal).toHaveBeenLastCalledWith({
              channel,
              msgid,
              succeeded: false,
            });
          }
        });

        it('handles various whisper failure message types', () => {
          const messageTypes = [
            'whisper_invalid_login',
            'whisper_invalid_self',
            'whisper_limit_per_min',
            'whisper_limit_per_sec',
            'whisper_restricted_recipient',
          ];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(connection.commands.whisper.signal).toHaveBeenLastCalledWith(
              {
                channel,
                msgid,
                succeeded: false,
              },
            );
          }
        });

        it('handles various permission errors', () => {
          const messageTypes = ['no_permission', 'msg_banned'];

          for (const msgid of messageTypes) {
            const { channel, connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, { body: '', channel, msgid });

            expect(connection.commands.failAll).toHaveBeenLastCalledWith(
              msgid,
              channel,
            );
          }
        });

        it('handles an "unrecognized_cmd" message type', () => {
          const msgid = 'unrecognized_cmd';
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
          });

          processor.processMessage(message);

          expectNoticeEventRaised(events, { body: '', channel, msgid });

          expect(connection.commands.failAll).toHaveBeenLastCalledWith(
            msgid,
            channel,
          );
        });

        it('handles "host_on" and "host_off" as no-ops', () => {
          for (const msgid of ['host_on', 'host_off']) {
            const { connection, events, message, processor } = setup({
              ...defaults,
              msgid,
            });

            processor.processMessage(message);

            expectNoop(events, connection);
          }
        });

        describe('handling when authentication is unsuccessful', () => {
          it('message contains "Login unsuccessful" or "Login authentication failed"', () => {
            for (const content of [
              ':Login unsuccessful',
              ':Login authentication failed',
            ]) {
              const msgid = 'foobar_unmatched';
              const { channel, connection, message, processor } = setup({
                ...defaults,
                msgid,
                params: [content],
              });

              processor.processMessage(message);

              // connect command is signaled
              expect(
                connection.commands.connect.signal,
              ).toHaveBeenLastCalledWith({
                channel,
                msgid,
                succeeded: false,
              });

              // disconnect is called
              expect(connection.disconnect).toHaveBeenLastCalledWith(false);
            }
          });

          it('messages to only disconnect for', () => {
            const loginMessages = [
              ':Error logging in',
              ':Improperly formatted auth',
              ':Invalid NICK',
            ];

            for (const loginMessage of loginMessages) {
              const msgid = 'foobar_unmatched';
              const { connection, message, processor } = setup({
                ...defaults,
                msgid,
                params: [loginMessage],
              });

              processor.processMessage(message);

              expect(connection.disconnect).toHaveBeenLastCalledWith(false);
            }
          });

          it('other messages', () => {
            const msgid = 'foobar_unmatched';
            const { channel, events, message, processor } = setup({
              ...defaults,
              msgid,
              params: [':Something went wrong'],
            });

            processor.processMessage(message);

            expectNoticeEventRaised(events, {
              body: 'Something went wrong',
              channel,
              msgid,
            });
          });
        });
      });

      describe('handling a USERNOTICE command', () => {
        beforeEach(() => {
          defaults.command = 'USERNOTICE';
        });

        it('handles a "resub" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'resub',
            tags: [
              'msg-param-sub-plan=TwitchPrime',
              'msg-param-sub-plan-name=foo',
              'msg-param-months=4',
              'msg-param-cumulative-months=4',
              'msg-param-streak-months=1',
              'msg-param-should-share-streak=false',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
              ...createSubscriptionFunFactFollowingAgeTags(),
            ],
          });

          processor.processMessage(message);

          expect(events.resub).toHaveBeenLastCalledWith({
            body: '',
            channel,
            cumulativeMonths: 4,
            funFact: mockSubscriptionFunFactFollowingAge,
            giftData: undefined,
            goalData: mockGoalData,
            methods: {
              plan: 'TwitchPrime',
              planName: 'foo',
              prime: true,
            },
            months: 4,
            shouldShareStreakTenure: false,
            streakMonths: 1,
            user: mockLoggedInUser,
            wasGifted: false,
          });
        });

        it('handles a "resub" message type with emote fun fact', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'resub',
            tags: [
              'msg-param-sub-plan=TwitchPrime',
              'msg-param-sub-plan-name=foo',
              'msg-param-months=4',
              'msg-param-cumulative-months=4',
              'msg-param-streak-months=1',
              'msg-param-should-share-streak=false',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
              ...createSubscriptionFunFactMostUsedEmoteTags(),
            ],
          });

          processor.processMessage(message);

          expect(events.resub).toHaveBeenLastCalledWith({
            body: '',
            channel,
            cumulativeMonths: 4,
            funFact: mockSubscriptionFunFactMostUsedEmote,
            giftData: undefined,
            goalData: mockGoalData,
            methods: {
              plan: 'TwitchPrime',
              planName: 'foo',
              prime: true,
            },
            months: 4,
            shouldShareStreakTenure: false,
            streakMonths: 1,
            user: mockLoggedInUser,
            wasGifted: false,
          });
        });

        it('handles a "resub" message type with gift info', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'resub',
            tags: [
              'msg-param-sub-plan=TwitchPrime',
              'msg-param-sub-plan-name=foo',
              'msg-param-months=4',
              'msg-param-cumulative-months=4',
              'msg-param-streak-months=1',
              'msg-param-should-share-streak=false',
              'msg-param-was-gifted=true',
              'msg-param-anon-gift=true',
              'msg-param-gifter-id=anon',
              'msg-param-gifter-login=anon2',
              'msg-param-gifter-name=anon3',
              'msg-param-gift-months=10',
              'msg-param-gift-month-being-redeemed=2',
              ...createUserTags(mockLoggedInUser),
              ...createSubscriptionFunFactMostUsedEmoteTags(),
            ],
          });

          processor.processMessage(message);

          expect(events.resub).toHaveBeenLastCalledWith({
            body: '',
            channel,
            cumulativeMonths: 4,
            funFact: mockSubscriptionFunFactMostUsedEmote,
            giftData: {
              anonGift: true,
              giftMonthBeingRedeemed: 2,
              giftedMonths: 10,
              gifterId: 'anon',
              gifterLogin: 'anon2',
              gifterName: 'anon3',
            },
            methods: {
              plan: 'TwitchPrime',
              planName: 'foo',
              prime: true,
            },
            months: 4,
            shouldShareStreakTenure: false,
            streakMonths: 1,
            user: mockLoggedInUser,
            wasGifted: true,
          });
        });

        it('handles a "resub" message type with multimonth info', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'resub',
            tags: [
              'msg-param-sub-plan=TwitchPrime',
              'msg-param-sub-plan-name=foo',
              'msg-param-months=4',
              'msg-param-cumulative-months=4',
              'msg-param-streak-months=1',
              'msg-param-should-share-streak=false',
              'msg-param-multimonth-tenure=2',
              'msg-param-multimonth-duration=3',
              ...createUserTags(mockLoggedInUser),
              ...createSubscriptionFunFactMostUsedEmoteTags(),
            ],
          });

          processor.processMessage(message);

          expect(events.resub).toHaveBeenLastCalledWith({
            body: '',
            channel,
            cumulativeMonths: 4,
            funFact: mockSubscriptionFunFactMostUsedEmote,
            giftData: undefined,
            methods: {
              plan: 'TwitchPrime',
              planName: 'foo',
              prime: true,
            },
            months: 4,
            multiMonthData: {
              multiMonthDuration: 3,
              multiMonthTenure: 2,
            },
            shouldShareStreakTenure: false,
            streakMonths: 1,
            user: mockLoggedInUser,
            wasGifted: false,
          });
        });

        it('handles a "extendsub" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'extendsub',
            tags: [
              'msg-param-sub-plan=1000',
              'msg-param-sub-benefit-end-month=October',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.extendsub).toHaveBeenLastCalledWith({
            benefitEndMonth: 'October',
            body: '',
            channel,
            goalData: mockGoalData,
            tier: '1000',
            user: mockLoggedInUser,
          });
        });

        it('handles a "communitypayforward" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'communitypayforward',
            tags: [
              'msg-param-prior-gifter-id=203316457',
              'msg-param-prior-gifter-display-name=camhux',
              'msg-param-prior-gifter-anonymous=true',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.communitypayforward).toHaveBeenLastCalledWith({
            channel,
            priorGifterAnonymous: true,
            priorGifterID: '203316457',
            priorGifterName: 'camhux',
            user: mockLoggedInUser,
          });
        });

        it('handles a "standardpayforward" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'standardpayforward',
            tags: [
              'msg-param-prior-gifter-id=203316457',
              'msg-param-prior-gifter-display-name=camhux',
              'msg-param-prior-gifter-anonymous=true',
              'msg-param-recipient-id=31643150',
              'msg-param-recipient-display-name=aWarpy',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.standardpayforward).toHaveBeenLastCalledWith({
            channel,
            priorGifterAnonymous: true,
            priorGifterID: '203316457',
            priorGifterName: 'camhux',
            recipientID: '31643150',
            recipientName: 'aWarpy',
            user: mockLoggedInUser,
          });
        });

        it('handles a "giftpaidupgrade" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'giftpaidupgrade',
            tags: [
              'msg-param-promo-gift-total=121',
              'msg-param-promo-name=Subtember',
              'msg-param-sender-login=daughnat',
              'msg-param-sender-name=daughnat',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.giftpaidupgrade).toHaveBeenLastCalledWith({
            channel,
            goalData: mockGoalData,
            promoGiftTotal: 121,
            promoName: 'Subtember',
            senderLogin: 'daughnat',
            senderName: 'daughnat',
            user: mockLoggedInUser,
          });
        });

        it('handles a "anongiftpaidupgrade" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'anongiftpaidupgrade',
            tags: [
              'msg-param-promo-gift-total=121',
              'msg-param-promo-name=Subtember',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.anongiftpaidupgrade).toHaveBeenLastCalledWith({
            channel,
            goalData: mockGoalData,
            promoGiftTotal: 121,
            promoName: 'Subtember',
            user: mockLoggedInUser,
          });
        });

        it('handles a "primepaidupgrade" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'primepaidupgrade',
            tags: [
              'msg-param-sub-plan=1000',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.primepaidupgrade).toHaveBeenLastCalledWith({
            channel,
            goalData: mockGoalData,
            plan: '1000',
            user: mockLoggedInUser,
          });
        });

        it('handles a "primecommunitygiftreceived" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'primecommunitygiftreceived',
            tags: [
              'msg-param-middle-man=ninja',
              'msg-param-gift-name=ninjaSweetLoot',
              'msg-param-sender=someRandUser',
              'msg-param-recipient=luckOtherRandUser',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.primecommunitygiftreceived).toHaveBeenLastCalledWith({
            channel,
            channelName: 'ninja',
            giftName: 'ninjaSweetLoot',
            receiver: 'luckOtherRandUser',
            sender: 'someRandUser',
            user: mockLoggedInUser,
          });
        });

        it('handles a "sub" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'sub',
            tags: [
              'msg-param-sub-plan=TwitchPrime',
              'msg-param-sub-plan-name=foo',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.subscription).toHaveBeenLastCalledWith({
            channel,
            goalData: mockGoalData,
            methods: {
              plan: 'TwitchPrime',
              planName: 'foo',
              prime: true,
            },
            user: mockLoggedInUser,
          });
        });

        it('handles a "sub" message type with multimonth data', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'sub',
            tags: [
              'msg-param-sub-plan=TwitchPrime',
              'msg-param-sub-plan-name=foo',
              'msg-param-multimonth-tenure=2',
              'msg-param-multimonth-duration=3',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.subscription).toHaveBeenLastCalledWith({
            channel,
            methods: {
              plan: 'TwitchPrime',
              planName: 'foo',
              prime: true,
            },
            multiMonthData: {
              multiMonthDuration: 3,
              multiMonthTenure: 2,
            },
            user: mockLoggedInUser,
          });
        });

        it('handles a "subgift" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'subgift',
            tags: [
              'msg-param-recipient-display-name=thedoc',
              'msg-param-recipient-user-name=thedoc',
              'msg-param-recipient-id=5678',
              'msg-param-sub-plan=TwitchPrime',
              'msg-param-sub-plan-name=foo',
              'msg-param-sender-count=10',
              'msg-param-gift-months=1',
              'msg-param-gift-theme=biblethump',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.subgift).toHaveBeenLastCalledWith({
            channel,
            giftMonths: 1,
            giftTheme: 'biblethump',
            goalData: mockGoalData,
            methods: {
              plan: 'TwitchPrime',
              planName: 'foo',
              prime: true,
            },
            recipientID: '5678',
            recipientLogin: 'thedoc',
            recipientName: 'thedoc',
            senderCount: 10,
            user: mockLoggedInUser,
          });
        });

        it('handles a "anonsubgift" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'anonsubgift',
            tags: [
              'msg-param-recipient-display-name=thedoc',
              'msg-param-recipient-user-name=thedoc',
              'msg-param-recipient-id=5678',
              'msg-param-sub-plan=TwitchPrime',
              'msg-param-sub-plan-name=foo',
              'msg-param-fun-string=lololol',
              'msg-param-gift-months=1',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.anonsubgift).toHaveBeenLastCalledWith({
            channel,
            funString: 'lololol',
            giftMonths: 1,
            goalData: mockGoalData,
            methods: {
              plan: 'TwitchPrime',
              planName: 'foo',
              prime: true,
            },
            recipientID: '5678',
            recipientLogin: 'thedoc',
            recipientName: 'thedoc',
          });
        });

        it('handles a "submysterygift" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'submysterygift',
            tags: [
              'msg-param-mass-gift-count=10',
              'msg-param-sub-plan=ponde',
              'msg-param-sender-count=20',
              'msg-param-gift-theme=biblethump',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.submysterygift).toHaveBeenLastCalledWith({
            channel,
            giftTheme: 'biblethump',
            goalData: mockGoalData,
            massGiftCount: 10,
            plan: 'ponde',
            senderCount: 20,
            user: mockLoggedInUser,
          });
        });

        it('handles a "anonsubmysterygift" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'anonsubmysterygift',
            tags: [
              'msg-param-mass-gift-count=10',
              'msg-param-sub-plan=ponde',
              'msg-param-fun-string=lololol',
              ...createGoalTags(mockGoalData),
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.anonsubmysterygift).toHaveBeenLastCalledWith({
            channel,
            funString: 'lololol',
            goalData: mockGoalData,
            massGiftCount: 10,
            plan: 'ponde',
          });
        });

        it('handles a "charity" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'charity',
            tags: [
              'msg-param-charity-name=mycharity',
              'msg-param-total=500',
              'msg-param-charity-days-remaining=3',
              'msg-param-charity-hours-remaining=8',
              'msg-param-charity-hashtag=#mycharity',
              'msg-param-charity-learn-more=foo',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.charity).toHaveBeenLastCalledWith({
            channel,
            charityName: 'mycharity',
            daysLeft: 3,
            hashtag: '#mycharity',
            hoursLeft: 8,
            learnMore: 'foo',
            total: 500,
          });
        });

        it('handles an "unraid" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'unraid',
            tags: [
              `display-name=${mockLoggedInUser.displayName}`,
              'system-msg=unraid',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.unraid).toHaveBeenLastCalledWith({
            channel,
            message: 'unraid',
            userLogin: mockLoggedInUser.displayName,
          });
        });

        it('handles a "raid" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'raid',
            tags: [
              `display-name=${mockLoggedInUser.displayName}`,
              `msg-param-displayName=${mockLoggedInUser.displayName}`,
              'msg-param-viewerCount=50',
              `msg-param-login=${mockLoggedInUser.username}`,
            ],
          });

          processor.processMessage(message);

          expect(events.raid).toHaveBeenLastCalledWith({
            channel,
            params: {
              displayName: mockLoggedInUser.displayName,
              login: mockLoggedInUser.username,
              msgId: 'raid',
              userID: '',
              viewerCount: '50',
            },
            userLogin: mockLoggedInUser.username,
          });
        });

        it('handles a "crate" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'crate',
            tags: [
              'msg-param-selectedCount=5',
              `display-name=${mockLoggedInUser.displayName}`,
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.crate).toHaveBeenLastCalledWith({
            channel,
            message: {
              body: '',
              id: '1234',
              timestamp: 0,
              user: mockLoggedInUser,
            },
            selectedCount: 5,
            timestamp: Date.now(),
            type: TMIChatEventType.Crate,
          });
        });

        it('handles a "rewardgift" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'rewardgift',
            tags: [
              'msg-param-trigger-type=bits',
              'msg-param-trigger-amount=500',
              'msg-param-selected-count=2',
              'msg-param-total-reward-count=4',
              'msg-param-domain=testdomain',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.rewardgift).toHaveBeenLastCalledWith({
            channel,
            domain: 'testdomain',
            message: {
              body: '',
              id: '1234',
              timestamp: 0,
              user: mockLoggedInUser,
            },
            selectedCount: 2,
            timestamp: Date.now(),
            totalRewardCount: 4,
            triggerAmount: 500,
            triggerType: 'bits',
            type: TMIChatEventType.RewardGift,
            user: mockLoggedInUser,
          });
        });

        it('handles a "purchase" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'purchase',
            tags: [
              'msg-param-title=mytitle',
              'msg-param-imageURL=myimageurl',
              'msg-param-crateCount=5',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.purchase).toHaveBeenLastCalledWith({
            channel,
            message: {
              body: '',
              id: '1234',
              timestamp: 0,
              user: mockLoggedInUser,
            },
            purchase: {
              crateLoot: [],
              numCrates: 5,
              purchased: {
                boxart: 'myimageurl',
                title: 'mytitle',
                type: 'game',
              },
            },
            timestamp: Date.now(),
            type: TMIChatEventType.Purchase,
          });
        });

        it('handles a "ritual" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'ritual',
            tags: [
              'msg-param-ritual-name=myritual',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.ritual).toHaveBeenLastCalledWith({
            channel,
            message: {
              body: '',
              id: '1234',
              timestamp: 0,
              user: mockLoggedInUser,
            },
            type: 'myritual',
          });
        });

        it('handles a "firstcheer" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'firstcheer',
            params: [':hello there'],
            tags: [...createUserTags(mockLoggedInUser)],
          });

          processor.processMessage(message);

          expect(events.firstcheer).toHaveBeenLastCalledWith({
            channel,
            message: {
              body: 'hello there',
              id: '1234',
              timestamp: 0,
              user: mockLoggedInUser,
            },
            sentByCurrentUser: false,
            timestamp: Date.now(),
            type: TMIChatEventType.FirstCheer,
          });
        });

        it('handles an "anoncheer" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'anoncheer',
            params: [':cheer1337'],
            tags: [...createUserTags(mockLoggedInUser)],
          });

          processor.processMessage(message);

          expect(events.anoncheer).toHaveBeenLastCalledWith({
            channel,
            message: {
              body: 'cheer1337',
              id: '1234',
              timestamp: 0,
              user: mockLoggedInUser,
            },
            sentByCurrentUser: false,
            timestamp: Date.now(),
            type: TMIChatEventType.AnonCheer,
          });
        });

        it('handles a "bitsbadgetier" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'bitsbadgetier',
            params: [':hello there'],
            tags: [
              'msg-param-threshold=100',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.bitsbadgetier).toHaveBeenLastCalledWith({
            channel,
            message: {
              body: 'hello there',
              id: '1234',
              timestamp: 0,
              user: mockLoggedInUser,
            },
            sentByCurrentUser: false,
            threshold: 100,
            timestamp: Date.now(),
            type: TMIChatEventType.BitsBadgeTier,
          });
        });

        it('handles a "contributechannelchallenge" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'contributechannelchallenge',
            params: [':hello there'],
            tags: [
              'msg-param-bits=100',
              'msg-param-userID=123',
              'msg-param-title=123',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.contributechannelchallenge).toHaveBeenLastCalledWith({
            bits: 100,
            channel,
            message: {
              body: 'hello there',
              id: '1234',
              timestamp: 0,
              user: mockLoggedInUser,
            },
            sentByCurrentUser: false,
            timestamp: Date.now(),
            title: '123',
            type: TMIChatEventType.ContributeChannelChallenge,
            userID: '123',
          });
        });

        it('handles a "celebrationpurchase" message type', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            msgid: 'celebrationpurchase',
            tags: [
              'msg-param-intensity=small',
              'msg-param-effect=flamethrower',
              ...createUserTags(mockLoggedInUser),
            ],
          });

          processor.processMessage(message);

          expect(events.celebrationpurchase).toHaveBeenLastCalledWith({
            channel,
            effect: 'flamethrower',
            intensity: 'small',
            user: mockLoggedInUser,
          });
        });
      });

      it('handles a "useranniversary" message type', () => {
        const { channel, events, message, processor } = setup({
          ...defaults,
          msgid: 'useranniversary',
          tags: ['msg-param-years=5', ...createUserTags(mockLoggedInUser)],
        });

        processor.processMessage(message);

        expect(events.useranniversary).toHaveBeenLastCalledWith({
          channel,
          message: {
            body: '',
            id: '1234',
            timestamp: 0,
            user: mockLoggedInUser,
          },
          years: 5,
        });
      });

      it('handles a "communityintroduction" message type', () => {
        const { channel, events, message, processor } = setup({
          ...defaults,
          msgid: 'communityintroduction',
          tags: createUserTags(mockLoggedInUser),
        });

        processor.processMessage(message);

        expect(events.communityintroduction).toHaveBeenLastCalledWith({
          channel,
          message: {
            body: '',
            id: '1234',
            timestamp: 0,
            user: mockLoggedInUser,
          },
        });
      });

      describe('handling a HOSTTARGET command', () => {
        beforeEach(() => {
          defaults.command = 'HOSTTARGET';
        });

        it('unhosts correctly', () => {
          const msgid = 'foo';
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
            params: ['-'],
          });

          processor.processMessage(message);

          expect(connection.commands.unhost.signal).toHaveBeenLastCalledWith({
            channel,
            msgid,
            succeeded: true,
          });

          expect(events.unhost).toHaveBeenLastCalledWith({
            channel,
            viewers: 0,
          });
        });

        it('hosts correctly', () => {
          const msgid = 'foo';
          const { channel, connection, events, message, processor } = setup({
            ...defaults,
            msgid,
            params: [':thedoc', '20'],
          });

          processor.processMessage(message);

          expect(connection.commands.host.signal).toHaveBeenLastCalledWith({
            channel,
            msgid,
            succeeded: true,
          });

          expect(events.hosting).toHaveBeenLastCalledWith({
            channel,
            target: 'thedoc',
            viewers: 20,
          });
        });
      });

      describe('handling a CLEARCHAT command', () => {
        beforeEach(() => {
          defaults.command = 'CLEARCHAT';
        });

        it('user was banned by a moderator', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            params: ['shrood'],
            tags: [
              // No 'ban-duration' tag for a ban vs. a timeout
              'ban-reason=toxic',
            ],
          });

          processor.processMessage(message);

          expect(events.ban).toHaveBeenLastCalledWith({
            channel,
            duration: null,
            reason: 'toxic',
            userLogin: 'shrood',
          });
        });

        it('user was timed out by a moderator', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            params: ['shrood'],
            tags: ['ban-duration=1000', 'ban-reason=spam'],
          });

          processor.processMessage(message);

          expect(events.timeout).toHaveBeenLastCalledWith({
            channel,
            duration: 1000,
            reason: 'spam',
            userLogin: 'shrood',
          });
        });

        it('chat was cleared', () => {
          const { channel, connection, events, message, processor } =
            setup(defaults);

          processor.processMessage(message);

          expect(events.clearchat).toHaveBeenLastCalledWith({
            channel,
          });

          expect(connection.commands.clearChat.signal).toHaveBeenLastCalledWith(
            {
              channel,
              msgid: '',
              succeeded: true,
            },
          );
        });
      });

      it('handles a CLEARMSG command', () => {
        const { channel, events, message, processor } = setup({
          ...defaults,
          command: 'CLEARMSG',
          params: [':Hello, World!'],
          tags: ['login=3v', 'target-msg-id=some-id'],
        });

        processor.processMessage(message);

        expect(events.clearmsg).toHaveBeenLastCalledWith({
          body: 'Hello, World!',
          channel,
          targetMessageID: 'some-id',
          userLogin: '3v',
        });
      });

      it('handles a RECONNECT command', () => {
        const { connection, message, processor } = setup({
          ...defaults,
          command: 'RECONNECT',
        });

        processor.processMessage(message);

        expect(connection.onReconnect).toHaveBeenCalledTimes(1);
      });

      it('handles a SERVERCHANGE command', () => {
        const { connection, events, message, processor } = setup({
          ...defaults,
          command: 'SERVERCHANGE',
        });

        processor.processMessage(message);

        expectNoop(events, connection);
      });

      describe('handling a USERSTATE command', () => {
        beforeEach(() => {
          defaults.command = 'USERSTATE';
        });

        it('adds the user to the moderators list if necessary', () => {
          const { channel, message, processor, session } = setup({
            ...defaults,
            tags: [
              // Specifies the user should be added as a mod
              'user-type=mod',
            ],
          });

          jest.spyOn(session, 'addChannelModerator');

          processor.processMessage(message);

          expect(session.addChannelModerator).toHaveBeenCalledWith(
            channel,
            session.username,
          );
        });

        it('joins the user to the channel', () => {
          const { channel, events, message, processor, session } = setup({
            ...defaults,
            tags: createUserTags(mockLoggedInUser),
          });

          // This specific command takes the username from the session rather than
          // the 'username' message tag.
          session.username = mockLoggedInUser.username;
          jest.spyOn(session, 'onJoinedChannel');

          processor.processMessage(message);

          expect(session.onJoinedChannel).toHaveBeenLastCalledWith(
            channel,
            mockLoggedInUser,
          );
          expect(events.joined).toHaveBeenLastCalledWith({
            channel,
            gotUsername: true,
            username: mockLoggedInUser.username,
          });
        });

        it("updates the user's badges if they changed", () => {
          const { channel, events, message, processor, session } = setup({
            ...defaults,
            tags: [...createUserTags(mockLoggedInUser), 'badges=moderator/1'],
          });

          // This specific command takes the username from the session rather than
          // the 'username' message tag.
          session.username = mockLoggedInUser.username;
          jest.spyOn(session, 'updateBadges');

          processor.processMessage(message);

          expect(session.updateBadges).toHaveBeenLastCalledWith(
            channel,
            { moderator: '1' },
            {},
          );
          expect(events.badgesupdated).toHaveBeenLastCalledWith({
            badgeDynamicData: {},
            badges: { moderator: '1' },
            username: mockLoggedInUser.username,
          });
        });

        it("updates the user's badges if they changed and include extra data", () => {
          const { channel, events, message, processor, session } = setup({
            ...defaults,
            tags: [
              ...createUserTags(mockLoggedInUser),
              'badges=subscriber/3',
              'badge-info=subscriber/5',
            ],
          });

          // This specific command takes the username from the session rather than
          // the 'username' message tag.
          session.username = mockLoggedInUser.username;
          jest.spyOn(session, 'updateBadges');

          processor.processMessage(message);

          expect(session.updateBadges).toHaveBeenLastCalledWith(
            channel,
            { subscriber: '3' },
            { subscriber: '5' },
          );
          expect(events.badgesupdated).toHaveBeenLastCalledWith({
            badgeDynamicData: { subscriber: '5' },
            badges: { subscriber: '3' },
            username: mockLoggedInUser.username,
          });
        });

        it('updates the userstate', () => {
          const { channel, message, processor, session } = setup({
            ...defaults,
            tags: createUserTags(mockLoggedInUser),
          });

          session.username = mockLoggedInUser.username;
          jest.spyOn(session, 'updateUserState');

          processor.processMessage(message);

          expect(session.updateUserState).toHaveBeenLastCalledWith(
            channel,
            mockLoggedInUser,
          );
        });
      });

      it('handles a GLOBALUSERSTATE command to update the session.globaluserstate', () => {
        const { message, processor, session } = setup({
          ...defaults,
          command: 'GLOBALUSERSTATE',
          tags: createUserTags(mockLoggedInUser),
        });

        processor.processMessage(message);

        // Sets the session.globaluserstate property to the message tags
        expect(session.globaluserstate).toEqual({
          color: mockLoggedInUser.color,
          displayName: mockLoggedInUser.displayName,
          id: mockLoggedInUser.id,
          userID: mockLoggedInUser.userID,
          username: mockLoggedInUser.username,
        });
      });

      describe('handling a ROOMSTATE command', () => {
        beforeEach(() => {
          defaults.command = 'ROOMSTATE';
        });

        it('updates room state', () => {
          const { channel, events, message, processor, session } = setup({
            ...defaults,
            tags: ['slow=10', 'followers-only=20'],
          });

          jest
            .spyOn(session, 'getChannelState')
            .mockReturnValue(new TMIChannel(mockLoggedInUser));

          processor.processMessage(message);

          // Raises a roomstate event
          expect(events.roomstate).toHaveBeenLastCalledWith({
            channel,
            state: expect.objectContaining({
              followersOnly: true,
              followersOnlyRequirement: 20,
              slowMode: true,
              slowModeDuration: 10,
            }),
          });

          // enables slowmode
          expect(events.slowmode).toHaveBeenLastCalledWith({
            channel,
            enabled: true,
            length: 10,
          });

          // enables followers-only mode
          expect(events.followersonly).toHaveBeenLastCalledWith({
            channel,
            enabled: true,
            length: 20,
          });
        });
      });
    });

    describe('messages with prefix "jtv"', () => {
      beforeEach(() => {
        defaults.prefix = 'jtv';
      });

      describe('handling a MODE command', () => {
        beforeEach(() => {
          defaults.command = 'MODE';
        });

        it('adds the username to the moderators when the message is "+o"', () => {
          const { channel, events, message, processor, session } = setup({
            ...defaults,
            params: ['+o', 'thedoc'],
          });

          jest.spyOn(session, 'addChannelModerator');

          processor.processMessage(message);

          expect(session.addChannelModerator).toHaveBeenLastCalledWith(
            channel,
            'thedoc',
          );
          expect(events.mod).toHaveBeenLastCalledWith({
            channel,
            username: 'thedoc',
          });
        });

        it('removes the username from the moderators when the message is "-o"', () => {
          const { channel, events, message, processor, session } = setup({
            ...defaults,
            params: ['-o', 'thedoc'],
          });

          jest.spyOn(session, 'removeChannelModerator');

          processor.processMessage(message);

          expect(session.removeChannelModerator).toHaveBeenLastCalledWith(
            channel,
            'thedoc',
          );
          expect(events.unmod).toHaveBeenLastCalledWith({
            channel,
            username: 'thedoc',
          });
        });
      });
    });

    describe('messages with other prefixes', () => {
      beforeEach(() => {
        defaults.prefix = 'otherprefix';
      });

      it('handles a 353 command', () => {
        const { events, message, processor } = setup({
          ...defaults,
          // This is a message with a unique parameter order, where channel is
          // the third param instead of the first parameter.
          channel: null,
          command: '353',
          params: ['myuser', '=', '#testchannel', ':thedoc user1 user2 user3'],
        });

        processor.processMessage(message);

        expect(events.names).toHaveBeenLastCalledWith({
          channel: '#testchannel',
          names: ['thedoc', 'user1', 'user2', 'user3'],
        });
      });

      describe('handling a JOIN command', () => {
        beforeEach(() => {
          defaults.command = 'JOIN';
        });

        it('an anonymous user', () => {
          const { channel, events, message, processor, session } = setup({
            ...defaults,
            prefix: createPrefixWithUsername('justinfan1234'),
            tags: createUserTags(mockAnonymousUser),
          });

          session.username = mockAnonymousUser.username;
          jest.spyOn(session, 'onJoinedChannel');

          processor.processMessage(message);

          expect(session.onJoinedChannel).toHaveBeenLastCalledWith(
            channel,
            mockAnonymousUser,
          );
          expect(events.joined).toHaveBeenLastCalledWith({
            channel,
            gotUsername: true,
            username: mockAnonymousUser.username,
          });
        });

        it('a logged in user', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            prefix: createPrefixWithUsername('thedoc'),
          });

          processor.processMessage(message);

          expect(events.joined).toHaveBeenLastCalledWith({
            channel,
            gotUsername: false,
            username: 'thedoc',
          });
        });
      });

      describe('handling a PART command', () => {
        beforeEach(() => {
          defaults.command = 'PART';
          defaults.tags = createUserTags(mockLoggedInUser);
        });

        it('for the current user', () => {
          const { channel, connection, events, message, processor, session } =
            setup({
              ...defaults,
              prefix: createPrefixWithUsername('shrood'),
            });

          session.username = mockLoggedInUser.username;
          jest.spyOn(session, 'onPartedChannel');

          processor.processMessage(message);

          expect(session.onPartedChannel).toHaveBeenLastCalledWith(channel);
          expect(connection.commands.part.signal).toHaveBeenLastCalledWith({
            channel,
            msgid: '',
            succeeded: true,
          });

          expect(events.parted).toHaveBeenLastCalledWith({
            channel,
            isSelf: true,
            username: mockLoggedInUser.username,
          });
        });

        it('for a different user', () => {
          const { channel, events, message, processor, session } = setup({
            ...defaults,
            prefix: createPrefixWithUsername('thedoc'),
          });

          session.username = mockLoggedInUser.username;

          processor.processMessage(message);

          expect(events.parted).toHaveBeenLastCalledWith({
            channel,
            isSelf: false,
            username: 'thedoc',
          });
        });
      });

      it('handles a WHISPER command', () => {
        const { events, message, processor } = setup({
          ...defaults,
          command: 'WHISPER',
          params: [':hello there'],
          tags: createUserTags(mockLoggedInUser),
        });

        processor.processMessage(message);

        expect(events.whisper).toHaveBeenLastCalledWith({
          body: 'hello there',
          sender: mockLoggedInUser,
          sentByCurrentUser: false,
        });
      });

      describe('handling a PRIVMSG command', () => {
        beforeEach(() => {
          defaults.command = 'PRIVMSG';
        });

        it('hosting you with viewer count', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            params: [':thedoc is now hosting you for 15 viewers'],
            prefix: createPrefixWithUsername('jtv'),
          });

          processor.processMessage(message);

          expect(events.hosted).toHaveBeenLastCalledWith({
            channel,
            from: 'thedoc',
            isAuto: false,
            viewers: 15,
          });
        });

        it('hosting you with a name that starts with a number', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            params: [':1234thedoc is now hosting you for 15 viewers'],
            prefix: createPrefixWithUsername('jtv'),
          });

          processor.processMessage(message);

          expect(events.hosted).toHaveBeenLastCalledWith({
            channel,
            from: '1234thedoc',
            isAuto: false,
            viewers: 15,
          });
        });

        it('hosting you without viewer count', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            params: [':thedoc is now hosting you'],
            prefix: createPrefixWithUsername('jtv'),
          });

          processor.processMessage(message);

          expect(events.hosted).toHaveBeenLastCalledWith({
            channel,
            from: 'thedoc',
            isAuto: false,
            viewers: 0,
          });
        });

        it('a regular message', () => {
          const { channel, events, message, processor } = setup({
            ...defaults,
            params: [':hello there'],
            prefix: createPrefixWithUsername('shrood'),
          });

          processor.processMessage(message);

          expect(events.chat).toHaveBeenLastCalledWith({
            channel,
            message: {
              body: 'hello there',
              id: '1234',
              timestamp: 0,
              user: mockLoggedInUser,
            },
            sentByCurrentUser: false,
            timestamp: Date.now(),
            type: TMIChatEventType.Message,
          });
        });

        describe('bits image urls', () => {
          it("doesn't a bits url", () => {
            const { channel, events, message, processor } = setup({
              ...defaults,
              params: [':hello there'],
              prefix: createPrefixWithUsername('shrood'),
              tags: [...createUserTags(mockLoggedInUser)],
            });

            processor.processMessage(message);

            expect(events.chat).toHaveBeenLastCalledWith({
              channel,
              message: {
                bitsImageUrl: undefined,
                body: 'hello there',
                id: '1234',
                timestamp: 0,
                user: mockLoggedInUser,
              },
              sentByCurrentUser: false,
              timestamp: Date.now(),
              type: TMIChatEventType.Message,
            });
          });

          it('has a bits url', () => {
            const { channel, events, message, processor } = setup({
              ...defaults,
              params: [':hello there'],
              prefix: createPrefixWithUsername('shrood'),
              tags: [
                ...createUserTags(mockLoggedInUser),
                'bits-img-url=https://twitch.tv/image.jpeg',
              ],
            });

            processor.processMessage(message);

            expect(events.chat).toHaveBeenLastCalledWith({
              channel,
              message: {
                bitsImageUrl: 'https://twitch.tv/image.jpeg',
                body: 'hello there',
                id: '1234',
                timestamp: 0,
                user: mockLoggedInUser,
              },
              sentByCurrentUser: false,
              timestamp: Date.now(),
              type: TMIChatEventType.Message,
            });
          });
        });

        describe('isFirstMsg', () => {
          it('is undefined if IRC tag first-msg is not included', () => {
            const { channel, events, message, processor } = setup({
              ...defaults,
              params: [':hello there'],
              prefix: createPrefixWithUsername('shrood'),
              tags: [...createUserTags(mockLoggedInUser)],
            });

            processor.processMessage(message);

            expect(events.chat).toHaveBeenLastCalledWith({
              channel,
              message: {
                body: 'hello there',
                id: '1234',
                isFirstMsg: undefined,
                timestamp: 0,
                user: mockLoggedInUser,
              },
              sentByCurrentUser: false,
              timestamp: Date.now(),
              type: TMIChatEventType.Message,
            });
          });

          it('is true if IRC tag first-msg=1', () => {
            const { channel, events, message, processor } = setup({
              ...defaults,
              params: [':hello there'],
              prefix: createPrefixWithUsername('shrood'),
              tags: [...createUserTags(mockLoggedInUser), 'first-msg=1'],
            });

            processor.processMessage(message);

            expect(events.chat).toHaveBeenLastCalledWith({
              channel,
              message: {
                body: 'hello there',
                id: '1234',
                isFirstMsg: true,
                timestamp: 0,
                user: mockLoggedInUser,
              },
              sentByCurrentUser: false,
              timestamp: Date.now(),
              type: TMIChatEventType.Message,
            });
          });
        });

        describe('channel points rewards', () => {
          it('a custom reward redemption message', () => {
            const { channel, events, message, processor } = setup({
              ...defaults,
              params: [':hello there'],
              prefix: createPrefixWithUsername('shrood'),
              tags: [
                ...createUserTags(mockLoggedInUser),
                'custom-reward-id=a-custom-reward',
              ],
            });

            processor.processMessage(message);

            expect(events.channelpointsreward).toHaveBeenLastCalledWith({
              channel,
              message: {
                body: 'hello there',
                id: '1234',
                timestamp: 0,
                user: mockLoggedInUser,
              },
              rewardID: 'a-custom-reward',
            });
          });

          it('a highlighted message', () => {
            const { channel, events, message, processor } = setup({
              ...defaults,
              msgid: 'highlighted-message',
              params: [':hello there'],
              prefix: createPrefixWithUsername('shrood'),
              tags: [...createUserTags(mockLoggedInUser), 'room-id=123'],
            });

            processor.processMessage(message);

            expect(events.channelpointsreward).toHaveBeenLastCalledWith({
              channel,
              message: {
                body: 'hello there',
                id: '1234',
                timestamp: 0,
                user: mockLoggedInUser,
              },
              rewardID: '123:SEND_HIGHLIGHTED_MESSAGE',
            });
          });

          it('a subs-only bypass message', () => {
            const { channel, events, message, processor } = setup({
              ...defaults,
              msgid: 'skip-subs-mode-message',
              params: [':hello there'],
              prefix: createPrefixWithUsername('shrood'),
              tags: [...createUserTags(mockLoggedInUser), 'room-id=123'],
            });

            processor.processMessage(message);

            expect(events.channelpointsreward).toHaveBeenLastCalledWith({
              channel,
              message: {
                body: 'hello there',
                id: '1234',
                timestamp: 0,
                user: mockLoggedInUser,
              },
              rewardID: '123:SINGLE_MESSAGE_BYPASS_SUB_MODE',
            });
          });
        });

        describe('message flags', () => {
          it('a message with flags', () => {
            const { channel, events, message, processor } = setup({
              ...defaults,
              params: [':badword'],
              prefix: createPrefixWithUsername('shrood'),
              tags: [...createUserTags(mockLoggedInUser), 'flags=0-7:P.6'],
            });

            processor.processMessage(message);

            expect(events.chat).toHaveBeenLastCalledWith({
              channel,
              message: {
                body: 'badword',
                flags: [
                  {
                    categories: {
                      [TMIFlagCategory.Profanity]: true,
                    },
                    endIndex: 7,
                    startIndex: 0,
                  },
                ],
                id: '1234',
                timestamp: 0,
                user: mockLoggedInUser,
              },
              sentByCurrentUser: false,
              timestamp: Date.now(),
              type: TMIChatEventType.Message,
            });
          });

          it('a message with a bad flags tag', () => {
            const { channel, events, logger, message, processor } = setup({
              ...defaults,
              params: [':badword'],
              prefix: createPrefixWithUsername('shrood'),
              tags: [...createUserTags(mockLoggedInUser), 'flags=...'],
            });

            processor.processMessage(message);

            expect(logger.error).toHaveBeenCalledWith(
              expect.any(Error),
              'Failed to parse message flags',
              '...',
            );
            expect(events.chat).toHaveBeenLastCalledWith({
              channel,
              message: {
                body: 'badword',
                id: '1234',
                timestamp: 0,
                user: mockLoggedInUser,
              },
              sentByCurrentUser: false,
              timestamp: Date.now(),
              type: TMIChatEventType.Message,
            });
          });
        });
      });
    });
  });
});
