import { exhaustedCase } from 'tachyon-utils-ts';
import type { EventProcessors } from './event-processors';
import type {
  TMIBadgeLoot,
  TMIBitsLoot,
  TMIChatMessage,
  TMIChatMessageEvent,
  TMICrateLoot,
  TMIEmoteLoot,
  TMIFlag,
  TMIInGameContentLoot,
  TMILogger,
  TMIMessage,
  TMIPurchase,
  TMIRitualType,
  TMISession,
  TMISubscriptionFunFact,
  TMISubscriptionGoalData,
  TMIUser,
} from './models';
import { TMIChatEventType, TMICrateLootType, parseFlagsTag } from './models';
import type { TMICommands } from './tmi-commands';
import type { TMIConnection } from './tmi-connection';
import type { TMIParser } from './tmi-parser';
import { Utils, parseReplyTags, updateRoomStateWithTags } from './utils';

const NUM_VIEWERS_REGEX = / (\d+) viewers/;
// eslint-disable-next-line no-control-regex
const ACTION_MESSAGE_REGEX = /^\u0001ACTION ([^\u0001]+)\u0001$/;

export class MessageProcessor {
  private readonly parser: TMIParser;
  private readonly commands: TMICommands;
  private readonly events: EventProcessors;
  private readonly session: TMISession;
  private readonly connection: TMIConnection;
  private readonly logger: TMILogger;

  constructor(
    parser: TMIParser,
    eventProcessors: EventProcessors,
    session: TMISession,
    connection: TMIConnection,
  ) {
    this.connection = connection;
    this.session = session;
    this.parser = parser;
    this.commands = connection.commands;
    this.events = eventProcessors;
    this.logger = this.connection.logger;
  }

  public processMessage(data: string): void {
    const parts = data.split('\r\n');
    parts
      .filter((m) => !!m)
      .forEach((str) => {
        const parsedMessage = this.parser.msg(str);
        this.handleMessage(parsedMessage);
      });
  }

  private raiseNotice(channel: string, msgid: string, body: string) {
    this.events.notice({
      body,
      channel,
      msgid,
      timestamp: Date.now(),
      type: TMIChatEventType.Notice,
    });
  }

  private createUser(message: TMIMessage): TMIUser {
    let username =
      message.tags.username ||
      message.tags.login ||
      message.prefix?.split('!')[0];

    if (!username) {
      username = '';
      this.logger.warn(
        `Could not parse user from message: \n${JSON.stringify(
          message,
          null,
          4,
        )}`,
      );
    }

    let displayName = username;
    const displayNameTag = Utils.decodeTag(message.tags['display-name']);
    if (displayNameTag) {
      displayName = displayNameTag.trim();
    }
    const user: TMIUser = {
      badgeDynamicData: message.badgeDynamicData || null,
      badges: message.badges || null,
      bits: Utils.parseInt(message.tags.bits, 0),
      color: Utils.decodeTag(message.tags.color),
      displayName,
      emotes: message.emotes || {},
      id: Utils.decodeTag(message.tags.id),
      turbo: Utils.parseBool(message.tags.turbo, false),
      userID: Utils.decodeTag(message.tags['user-id']),
      userType: Utils.decodeTag(message.tags['user-type']),
      username,
    };

    return user;
  }

  private createChatMessageEvent(
    type: TMIChatEventType,
    channel: string,
    chatMessage: TMIChatMessage,
    sentByCurrentUser: boolean,
  ): TMIChatMessageEvent {
    return {
      channel,
      message: chatMessage,
      sentByCurrentUser,
      timestamp: Date.now(),
      type,
    };
  }

  private createChatMessage(message: TMIMessage, body: string): TMIChatMessage {
    let flags: Array<TMIFlag> | undefined;
    try {
      flags = parseFlagsTag(message.tags.flags);
    } catch (err) {
      this.logger.error(
        err,
        'Failed to parse message flags',
        message.tags.flags,
      );
    }
    const reply = parseReplyTags(message.tags);
    return {
      bitsImageUrl: message.tags['bits-img-url']
        ? Utils.decodeTag(message.tags['bits-img-url'])
        : undefined,
      body,
      flags,
      id: Utils.decodeTag(message.tags.id),
      isFirstMsg: Utils.parseBool(message.tags['first-msg'], undefined),
      nonce: message.tags.nonce,
      reply,
      timestamp: Utils.parseInt(message.tags['tmi-sent-ts'], 0),
      user: this.createUser(message),
    };
  }

  private getGoalData(
    message: TMIMessage,
  ): TMISubscriptionGoalData | undefined {
    const targetContributions = Utils.parseInt(
      message.tags['msg-param-goal-target-contributions'],
      0,
    );

    if (!targetContributions) {
      return;
    }
    return {
      contributionType: Utils.decodeTag(
        message.tags['msg-param-goal-contribution-type'],
      ),
      currentContributions: Utils.parseInt(
        message.tags['msg-param-goal-current-contributions'],
        0,
      ),
      description: Utils.decodeTag(message.tags['msg-param-goal-description']),
      targetContributions,
      userContributions: Utils.parseInt(
        message.tags['msg-param-goal-user-contributions'],
        0,
      ),
    };
  }

  private getSubscriptionFunFact(
    message: TMIMessage,
  ): TMISubscriptionFunFact | undefined {
    const funFactType = Utils.decodeTag(
      message.tags['msg-param-fun-fact-type'],
    ) as 'emote-usage' | 'following-age';
    if (!funFactType) {
      return;
    }

    switch (funFactType) {
      case 'following-age':
        return {
          followingAge: Utils.parseInt(
            message.tags['msg-param-fun-fact-content'],
            0,
          ),
          funFactType,
        };
      case 'emote-usage':
        return {
          funFactType,
          mostUsedEmoteID: Utils.decodeTag(
            message.tags['msg-param-fun-fact-content'],
          ),
        };
      default:
        return exhaustedCase(funFactType, undefined);
    }
  }

  private handleMessage(message: TMIMessage | null) {
    if (!message) {
      return;
    }

    const channel: string = Utils.channel(
      Utils.coalesce(message.params[0], ''),
    );

    // The raw content of the message
    const msg = Utils.coalesce(message.params[1], '');

    // The type of message, ex: 'followers_on'
    const msgid = Utils.coalesce(message.tags['msg-id'], '');

    // Parse badges and emotes.
    this.parser.badges(message);
    this.parser.emotes(message);

    // Messages with no prefix.
    if (!message.prefix) {
      switch (message.command) {
        // Received PING from server.
        case 'PING':
          this.connection.pong();
          break;

        // Received PONG from server, return current latency.
        case 'PONG':
          this.commands.ping.signal({ channel: undefined, msgid });
          break;

        default:
          this.logger.warn(
            `Could not parse message with no prefix:\n${JSON.stringify(
              message,
              null,
              4,
            )}`,
          );
          break;
      }
    } else if (message.prefix === 'tmi.twitch.tv') {
      // Messages with "tmi.twitch.tv" as a prefix.
      switch (message.command) {
        case '002':
        case '003':
        case '004':
        case '375':
        case '376':
        case 'CAP':
          break;

        // Retrieve username from server.
        case '001':
          this.session.username = message.params[0];
          break;

        // Connected to server.
        case '372':
          this.commands.connect.signal({ channel, msgid, succeeded: true });
          break;
        // https://github.com/justintv/Twitch-API/blob/master/chat/capabilities.md#notice
        case 'NOTICE':
          switch (msgid) {
            // This room is now in subscribers-only mode.
            case 'subs_on':
              this.commands.subscriberModeOn.signal({
                channel,
                msgid,
                succeeded: true,
              });
              this.events.subscribers({ channel, enabled: true });
              break;

            // This room is no longer in subscribers-only mode.
            case 'subs_off':
              this.commands.subscriberModeOff.signal({
                channel,
                msgid,
                succeeded: true,
              });
              this.events.subscribers({ channel, enabled: false });
              break;

            // This room is now in emote-only mode.
            case 'emote_only_on':
              this.events.emoteonlymode({ channel, enabled: true });
              this.commands.emoteOnlyModeOn.signal({ channel });
              break;

            // This room is no longer in emote-only mode.
            case 'emote_only_off':
              this.events.emoteonlymode({ channel, enabled: false });
              this.commands.emoteOnlyModeOff.signal({ channel });
              break;

            // Do not handle slow_on/off here, listen to the ROOMSTATE notice instead as it returns the delay.
            case 'slow_on':
              this.commands.slowModeOn.signal({
                channel,
                msgid,
                succeeded: true,
              });
              break;
            case 'slow_off':
              this.commands.slowModeOff.signal({
                channel,
                msgid,
                succeeded: true,
              });
              break;

            // Do not handle followers_on/off here, listen to the ROOMSTATE notice instead as it returns the delay.
            case 'followers_on_zero':
            case 'followers_on':
              this.commands.followersOnlyOn.signal({
                channel,
                msgid,
                succeeded: true,
              });
              break;
            case 'followers_off':
              this.commands.followersOnlyOff.signal({
                channel,
                msgid,
                succeeded: true,
              });
              break;

            // This room is now in r9k mode.
            case 'r9k_on':
              this.events.r9kmode({ channel, enabled: true });
              this.commands.rk9ModeOn.signal({ channel });
              break;

            // This room is no longer in r9k mode.
            case 'r9k_off':
              this.events.r9kmode({ channel, enabled: false });
              this.commands.rk9ModeOff.signal({ channel });
              break;

            // The moderators of this room are [...]
            case 'room_mods': {
              const splitted = msg.split(':');
              const mods = splitted[1]
                .replace(/,/g, '')
                .split(':')
                .toString()
                .toLowerCase()
                .split(' ');

              for (let i = mods.length - 1; i >= 0; i--) {
                if (mods[i] === '') {
                  mods.splice(i, 1);
                }
              }
              this.events.mods({ channel, usernames: mods });
              break;
            }

            // There are no moderators for this room.
            case 'no_mods':
              this.events.mods({ channel, usernames: [] });
              break;

            // Channel is suspended.
            case 'msg_channel_suspended':
              this.raiseNotice(channel, msgid, msg);
              break;

            // Ban command failed.
            case 'already_banned':
            case 'bad_ban_admin':
            case 'bad_ban_broadcaster':
            case 'bad_ban_global_mod':
            case 'bad_ban_self':
            case 'bad_ban_staff':
            case 'usage_ban':
              this.commands.ban.signal({ channel, msgid, succeeded: false });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Ban command success.
            case 'ban_success':
              this.commands.ban.signal({ channel, msgid, succeeded: true });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Clear command failed.
            case 'usage_clear':
              this.raiseNotice(channel, msgid, msg);
              break;

            // Mods command failed.
            case 'usage_mods':
              this.raiseNotice(channel, msgid, msg);
              // TODO: Implement /mods command
              break;

            // Mod command success.
            case 'mod_success':
              this.raiseNotice(channel, msgid, msg);
              // TODO: Implement /mods command
              break;

            // Mod command failed.
            case 'usage_mod':
            case 'bad_mod_banned':
            case 'bad_mod_mod':
              this.raiseNotice(channel, msgid, msg);
              break;

            // Unmod command success.
            case 'unmod_success':
              this.raiseNotice(channel, msgid, msg);
              break;

            // Unmod command failed.
            case 'usage_unmod':
            case 'bad_unmod_mod':
              this.raiseNotice(channel, msgid, msg);
              break;

            // Color command success.
            case 'color_changed':
              this.commands.color.signal({ channel, msgid, succeeded: true });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Color command failed.
            case 'usage_color':
            case 'turbo_only_color':
              this.commands.color.signal({ channel, msgid, succeeded: false });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Commercial command success.
            case 'commercial_success':
              this.commands.commercial.signal({
                channel,
                msgid,
                succeeded: true,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Commercial command failed.
            case 'usage_commercial':
            case 'bad_commercial_error':
              this.commands.commercial.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Host command success.
            case 'hosts_remaining': {
              this.commands.host.signal({
                channel,
                msgid,
                remainingHost: Utils.parseInt(msg.charAt(0), 0),
                succeeded: true,
              });
              this.raiseNotice(channel, msgid, msg);
              break;
            }

            // Host command failed.
            case 'bad_host_hosting':
            case 'bad_host_rate_exceeded':
            case 'bad_host_error':
            case 'usage_host':
              this.commands.host.signal({ channel, msgid, succeeded: false });
              this.raiseNotice(channel, msgid, msg);
              break;

            // r9kbeta command failed.
            case 'already_r9k_on':
            case 'usage_r9k_on':
              this.commands.rk9ModeOn.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // r9kbetaoff command failed.
            case 'already_r9k_off':
            case 'usage_r9k_off':
              this.commands.rk9ModeOff.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Timeout command success.
            case 'timeout_success':
              this.commands.timeout.signal({ channel, msgid, succeeded: true });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Subscribersoff command failed.
            case 'already_subs_off':
            case 'usage_subs_off':
              this.commands.subscriberModeOff.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Subscribers command failed.
            case 'already_subs_on':
            case 'usage_subs_on':
              this.commands.subscriberModeOn.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Emoteonlyoff command failed.
            case 'already_emote_only_off':
            case 'usage_emote_only_off':
              this.commands.emoteOnlyModeOff.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Emoteonly command failed.
            case 'already_emote_only_on':
            case 'usage_emote_only_on':
              this.commands.emoteOnlyModeOn.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Slow command failed.
            case 'usage_slow_on':
              this.commands.slowModeOn.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Slowoff command failed.
            case 'usage_slow_off':
              this.commands.slowModeOff.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Timeout command failed.
            case 'usage_timeout':
            case 'bad_timeout_admin':
            case 'bad_timeout_broadcaster':
            case 'bad_timeout_duration':
            case 'bad_timeout_global_mod':
            case 'bad_timeout_self':
            case 'bad_timeout_staff':
              this.commands.timeout.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Unban command success.
            case 'unban_success':
              this.commands.unban.signal({ channel, msgid, succeeded: true });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Unban command failed.
            case 'usage_unban':
            case 'bad_unban_no_ban':
              this.commands.unban.signal({ channel, msgid, succeeded: false });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Unhost command failed.
            case 'usage_unhost':
            case 'not_hosting':
              this.commands.unhost.signal({ channel, msgid, succeeded: false });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Whisper command failed.
            case 'whisper_invalid_login':
            case 'whisper_invalid_self':
            case 'whisper_limit_per_min':
            case 'whisper_limit_per_sec':
            case 'whisper_restricted_recipient':
              this.commands.whisper.signal({
                channel,
                msgid,
                succeeded: false,
              });
              this.raiseNotice(channel, msgid, msg);
              break;

            // Permission error.
            case 'no_permission':
            case 'msg_banned':
              this.logger.warn('No permission or banned', { channel, msg });
              // Shut them all down!
              this.commands.failAll(msgid, channel);
              this.raiseNotice(channel, msgid, msg);
              break;

            // Unrecognized command.
            case 'unrecognized_cmd':
              this.commands.failAll(msgid, channel);
              this.raiseNotice(channel, msgid, msg);
              if (msg.split(' ').splice(-1)[0] === '/w') {
                this.logger.warn(
                  'You must be connected to a group server to send or receive whispers.',
                );
              }
              break;

            // Ignore this because we are already listening to HOSTTARGET.
            case 'host_on':
            case 'host_off':
              break;

            default:
              if (
                msg.includes('Login unsuccessful') ||
                msg.includes('Login authentication failed')
              ) {
                this.commands.connect.signal({
                  channel,
                  msgid,
                  succeeded: false,
                });
                this.connection.disconnect(false);
              } else if (
                msg.includes('Error logging in') ||
                msg.includes('Improperly formatted auth')
              ) {
                this.connection.disconnect(false);
              } else if (msg.includes('Invalid NICK')) {
                this.connection.disconnect(false);
              } else if (msgid) {
                this.raiseNotice(channel, msgid, msg);
                this.logger.debug(
                  `Could not parse NOTICE from tmi.twitch.tv:\n${JSON.stringify(
                    message,
                    null,
                    4,
                  )}`,
                );
              } else {
                this.logger.warn(
                  `Could not parse NOTICE from tmi.twitch.tv:\n${JSON.stringify(
                    message,
                    null,
                    4,
                  )}`,
                );
              }
              break;
          }
          break;

        // Handle subanniversary / resub / gift / raid messages.
        case 'USERNOTICE':
          if (msgid === 'resub') {
            const plan = Utils.decodeTag(message.tags['msg-param-sub-plan']);
            const planName = Utils.decodeTag(
              message.tags['msg-param-sub-plan-name'],
            );
            const months = Utils.parseInt(message.tags['msg-param-months'], 0);
            const prime = plan.includes('Prime');
            const cumulativeMonths = Utils.parseInt(
              message.tags['msg-param-cumulative-months'],
              0,
            );
            const shouldShareStreakTenure = Utils.parseBool(
              message.tags['msg-param-should-share-streak'],
              false,
            );
            const streakMonths = Utils.parseInt(
              message.tags['msg-param-streak-months'],
              undefined,
            );
            const wasGifted = Utils.parseBool(
              message.tags['msg-param-was-gifted'],
              false,
            );

            if (msg) {
              message.tags['message-type'] = 'resub';
            }

            let giftData = undefined;
            if (wasGifted) {
              giftData = {
                anonGift: Utils.parseBool(
                  message.tags['msg-param-anon-gift'],
                  false,
                ),
                giftMonthBeingRedeemed: Utils.parseInt(
                  message.tags['msg-param-gift-month-being-redeemed'],
                  1,
                ),
                giftedMonths: Utils.parseInt(
                  message.tags['msg-param-gift-months'],
                  1,
                ),
                gifterId: Utils.decodeTag(message.tags['msg-param-gifter-id']),
                gifterLogin: Utils.decodeTag(
                  message.tags['msg-param-gifter-login'],
                ),
                gifterName: Utils.decodeTag(
                  message.tags['msg-param-gifter-name'],
                ),
              };
            }

            let multiMonthData = undefined;
            const multiMonthDuration = Utils.parseInt(
              message.tags['msg-param-multimonth-duration'],
              1,
            );
            if (multiMonthDuration > 1) {
              multiMonthData = {
                multiMonthDuration,
                multiMonthTenure: Utils.parseInt(
                  message.tags['msg-param-multimonth-tenure'],
                  1,
                ),
              };
            }

            this.events.resub({
              body: msg,
              channel,
              cumulativeMonths,
              funFact: this.getSubscriptionFunFact(message),
              giftData,
              goalData: this.getGoalData(message),
              methods: {
                plan,
                planName,
                prime,
              },
              months,
              multiMonthData,
              shouldShareStreakTenure,
              streakMonths,
              user: this.createUser(message),
              wasGifted,
            });
          } else if (msgid === 'extendsub') {
            const plan = Utils.decodeTag(message.tags['msg-param-sub-plan']);
            const benefitEndMonth = Utils.decodeTag(
              message.tags['msg-param-sub-benefit-end-month'],
            );

            this.events.extendsub({
              benefitEndMonth,
              body: msg,
              channel,
              goalData: this.getGoalData(message),
              tier: plan,
              user: this.createUser(message),
            });
          } else if (msgid === 'giftpaidupgrade') {
            this.events.giftpaidupgrade({
              channel,
              goalData: this.getGoalData(message),
              promoGiftTotal: Utils.parseInt(
                message.tags['msg-param-promo-gift-total'],
                0,
              ),
              promoName: Utils.decodeTag(message.tags['msg-param-promo-name']),
              senderLogin: Utils.decodeTag(
                message.tags['msg-param-sender-login'],
              ),
              senderName: Utils.decodeTag(
                message.tags['msg-param-sender-name'],
              ),
              user: this.createUser(message),
            });
          } else if (msgid === 'anongiftpaidupgrade') {
            this.events.anongiftpaidupgrade({
              channel,
              goalData: this.getGoalData(message),
              promoGiftTotal: Utils.parseInt(
                message.tags['msg-param-promo-gift-total'],
                0,
              ),
              promoName: Utils.decodeTag(message.tags['msg-param-promo-name']),
              user: this.createUser(message),
            });
          } else if (msgid === 'primepaidupgrade') {
            this.events.primepaidupgrade({
              channel,
              goalData: this.getGoalData(message),
              plan: Utils.decodeTag(message.tags['msg-param-sub-plan']),
              user: this.createUser(message),
            });
          } else if (msgid === 'primecommunitygiftreceived') {
            this.events.primecommunitygiftreceived({
              channel,
              channelName: Utils.decodeTag(
                message.tags['msg-param-middle-man'],
              ),
              giftName: Utils.decodeTag(message.tags['msg-param-gift-name']),
              receiver: Utils.decodeTag(message.tags['msg-param-recipient']),
              sender: Utils.decodeTag(message.tags['msg-param-sender']),
              user: this.createUser(message),
            });
          } else if (msgid === 'sub') {
            const plan = Utils.decodeTag(message.tags['msg-param-sub-plan']);
            const planName = Utils.decodeTag(
              message.tags['msg-param-sub-plan-name'],
            );
            const prime = plan.includes('Prime');

            if (msg) {
              message.tags['message-type'] = 'sub';
            }

            let multiMonthData = undefined;
            const multiMonthDuration = Utils.parseInt(
              message.tags['msg-param-multimonth-duration'],
              1,
            );
            if (multiMonthDuration > 1) {
              multiMonthData = {
                multiMonthDuration,
                multiMonthTenure: Utils.parseInt(
                  message.tags['msg-param-multimonth-tenure'],
                  1,
                ),
              };
            }

            this.events.subscription({
              channel,
              goalData: this.getGoalData(message),
              methods: {
                plan,
                planName,
                prime,
              },
              multiMonthData,
              user: this.createUser(message),
            });
          } else if (msgid === 'subgift') {
            const plan = Utils.decodeTag(message.tags['msg-param-sub-plan']);

            // Prime and msg should never be true in the current implementation, leaving the checks for future-proofing
            const prime = plan.includes('Prime');

            if (msg) {
              message.tags['message-type'] = 'subgift';
            }

            this.events.subgift({
              channel,
              giftMonths: Utils.parseInt(
                message.tags['msg-param-gift-months'],
                1,
              ),
              giftTheme: Utils.decodeTag(message.tags['msg-param-gift-theme']),
              goalData: this.getGoalData(message),
              methods: {
                plan,
                planName: Utils.decodeTag(
                  message.tags['msg-param-sub-plan-name'],
                ),
                prime,
              },
              recipientID: Utils.decodeTag(
                message.tags['msg-param-recipient-id'],
              ),
              recipientLogin: Utils.decodeTag(
                message.tags['msg-param-recipient-user-name'],
              ),
              recipientName: Utils.decodeTag(
                message.tags['msg-param-recipient-display-name'],
              ),
              senderCount: Utils.parseInt(
                message.tags['msg-param-sender-count'],
                0,
              ),
              user: this.createUser(message),
            });
          } else if (msgid === 'anonsubgift') {
            const plan = Utils.decodeTag(message.tags['msg-param-sub-plan']);

            // Prime and msg should never be true in the current implementation, leaving the checks for future-proofing
            const prime = plan.includes('Prime');

            if (msg) {
              message.tags['message-type'] = 'anonsubgift';
            }

            this.events.anonsubgift({
              channel,
              funString: Utils.decodeTag(message.tags['msg-param-fun-string']),
              giftMonths: Utils.parseInt(
                message.tags['msg-param-gift-months'],
                1,
              ),
              goalData: this.getGoalData(message),
              methods: {
                plan,
                planName: Utils.decodeTag(
                  message.tags['msg-param-sub-plan-name'],
                ),
                prime,
              },
              recipientID: Utils.decodeTag(
                message.tags['msg-param-recipient-id'],
              ),
              recipientLogin: Utils.decodeTag(
                message.tags['msg-param-recipient-user-name'],
              ),
              recipientName: Utils.decodeTag(
                message.tags['msg-param-recipient-display-name'],
              ),
            });
          } else if (msgid === 'submysterygift') {
            this.events.submysterygift({
              channel,
              giftTheme: Utils.decodeTag(message.tags['msg-param-gift-theme']),
              goalData: this.getGoalData(message),
              massGiftCount: Utils.parseInt(
                message.tags['msg-param-mass-gift-count'],
                0,
              ),
              plan: Utils.decodeTag(message.tags['msg-param-sub-plan']),
              senderCount: Utils.parseInt(
                message.tags['msg-param-sender-count'],
                0,
              ),
              user: this.createUser(message),
            });
          } else if (msgid === 'anonsubmysterygift') {
            this.events.anonsubmysterygift({
              channel,
              funString: Utils.decodeTag(message.tags['msg-param-fun-string']),
              goalData: this.getGoalData(message),
              massGiftCount: Utils.parseInt(
                message.tags['msg-param-mass-gift-count'],
                0,
              ),
              plan: Utils.decodeTag(message.tags['msg-param-sub-plan']),
            });
          } else if (msgid === 'communitypayforward') {
            this.events.communitypayforward({
              channel,
              priorGifterAnonymous: Utils.parseBool(
                message.tags['msg-param-prior-gifter-anonymous'],
                false,
              ),
              priorGifterID: Utils.decodeTag(
                message.tags['msg-param-prior-gifter-id'],
              ),
              priorGifterName: Utils.decodeTag(
                message.tags['msg-param-prior-gifter-display-name'],
              ),
              user: this.createUser(message),
            });
          } else if (msgid === 'standardpayforward') {
            this.events.standardpayforward({
              channel,
              priorGifterAnonymous: Utils.parseBool(
                message.tags['msg-param-prior-gifter-anonymous'],
                false,
              ),
              priorGifterID: Utils.decodeTag(
                message.tags['msg-param-prior-gifter-id'],
              ),
              priorGifterName: Utils.decodeTag(
                message.tags['msg-param-prior-gifter-display-name'],
              ),
              recipientID: Utils.decodeTag(
                message.tags['msg-param-recipient-id'],
              ),
              recipientName: Utils.decodeTag(
                message.tags['msg-param-recipient-display-name'],
              ),
              user: this.createUser(message),
            });
          } else if (msgid === 'charity') {
            const charityName = message.tags['msg-param-charity-name'];
            const bitsTotal = Number(message.tags['msg-param-total']);
            const daysLeft = Number(
              message.tags['msg-param-charity-days-remaining'],
            );
            const hoursLeft = Number(
              message.tags['msg-param-charity-hours-remaining'],
            );
            const hashtag = message.tags['msg-param-charity-hashtag'];
            const learnMore = message.tags['msg-param-charity-learn-more'];

            if (
              isNaN(bitsTotal) ||
              isNaN(daysLeft) ||
              isNaN(hoursLeft) ||
              !charityName ||
              !hashtag ||
              !learnMore
            ) {
              break;
            }

            this.events.charity({
              channel,
              charityName: Utils.decodeTag(charityName),
              daysLeft,
              hashtag,
              hoursLeft,
              learnMore,
              total: bitsTotal,
            });
          } else if (msgid === 'unraid') {
            const username =
              Utils.decodeTag(message.tags['display-name']) ||
              Utils.decodeTag(message.tags.login);
            // Just following the same parsing logic like in previous instances.
            const systemMessage = Utils.decodeTag(message.tags['system-msg']);
            this.events.unraid({
              channel,
              message: systemMessage,
              userLogin: username,
            });
          } else if (msgid === 'raid') {
            const username =
              Utils.decodeTag(message.tags['display-name']) ||
              Utils.decodeTag(message.tags.login);
            const displayName = Utils.decodeTag(
              message.tags['msg-param-displayName'],
            );
            const viewerCount = Utils.decodeTag(
              message.tags['msg-param-viewerCount'],
            );
            const login = Utils.decodeTag(message.tags['msg-param-login']);
            this.events.raid({
              channel,
              params: {
                displayName,
                login,
                msgId: msgid,
                userID: Utils.decodeTag(message.tags['user-id']),
                viewerCount,
              },
              userLogin: username,
            });
          } else if (msgid === 'crate') {
            const selectedCount = Utils.parseInt(
              message.tags['msg-param-selectedCount'],
              0,
            );
            const chatMessage = this.createChatMessage(message, msg);
            this.events.crate({
              channel,
              message: chatMessage,
              selectedCount,
              timestamp: Date.now(),
              type: TMIChatEventType.Crate,
            });
          } else if (msgid === 'rewardgift') {
            const chatMessage = this.createChatMessage(message, msg);
            const triggerType = Utils.decodeTag(
              message.tags['msg-param-trigger-type'],
            );
            const triggerAmount = Utils.parseInt(
              message.tags['msg-param-trigger-amount'],
              0,
            );
            const selectedCount = Utils.parseInt(
              message.tags['msg-param-selected-count'],
              0,
            );
            const totalRewardCount = Utils.parseInt(
              message.tags['msg-param-total-reward-count'],
              0,
            );
            const domain = Utils.decodeTag(message.tags['msg-param-domain']);
            this.events.rewardgift({
              channel,
              domain,
              message: chatMessage,
              selectedCount,
              timestamp: Date.now(),
              totalRewardCount,
              triggerAmount,
              triggerType,
              type: TMIChatEventType.RewardGift,
              user: this.createUser(message),
            });
          } else if (msgid === 'purchase') {
            const { tags } = message;
            const commerceRichContent: TMIPurchase = {
              crateLoot: [] as TMICrateLoot[],
              numCrates: Utils.parseInt(tags['msg-param-crateCount'], 0),
              purchased: {
                boxart: Utils.decodeTag(tags['msg-param-imageURL']),
                title: Utils.decodeTag(tags['msg-param-title']),
                type: 'game',
              },
            };
            const emotes = tags['msg-param-emoticons'];
            if (emotes) {
              commerceRichContent.crateLoot.push(
                ...emotes.split(',').map((id: string) => {
                  return { id, type: TMICrateLootType.Emote } as TMIEmoteLoot;
                }),
              );
            }

            const badges = tags['msg-param-badges'];
            if (badges) {
              commerceRichContent.crateLoot.push(
                ...badges.split(',').map((img: string) => {
                  return { img, type: TMICrateLootType.Badge } as TMIBadgeLoot;
                }),
              );
            }

            const bits = Utils.parseInt(tags['msg-param-bits'], 0);
            if (bits !== 0) {
              commerceRichContent.crateLoot.push({
                quantity: bits,
                type: TMICrateLootType.Bits,
              } as TMIBitsLoot);
            }

            const igc = tags['msg-param-inGameContent'];
            if (igc) {
              commerceRichContent.crateLoot.push(
                ...igc.split(',').map((img: string) => {
                  return {
                    img,
                    type: TMICrateLootType.InGameContent,
                  } as TMIInGameContentLoot;
                }),
              );
            }

            this.events.purchase({
              channel,
              message: this.createChatMessage(message, msg),
              purchase: commerceRichContent,
              timestamp: Date.now(),
              type: TMIChatEventType.Purchase,
            });
          } else if (msgid === 'ritual') {
            const ritualName = message.tags[
              'msg-param-ritual-name'
            ] as TMIRitualType;
            this.events.ritual({
              channel,
              message: this.createChatMessage(message, msg),
              type: ritualName,
            });
          } else if (msgid === 'firstcheer') {
            const chatMessage = this.createChatMessage(message, msg);
            this.events.firstcheer(
              this.createChatMessageEvent(
                TMIChatEventType.FirstCheer,
                channel,
                chatMessage,
                false,
              ),
            );
          } else if (msgid === 'anoncheer') {
            const chatMessage = this.createChatMessage(message, msg);
            this.events.anoncheer(
              this.createChatMessageEvent(
                TMIChatEventType.AnonCheer,
                channel,
                chatMessage,
                false,
              ),
            );
          } else if (msgid === 'bitsbadgetier') {
            const threshold = Utils.parseInt(
              message.tags['msg-param-threshold'],
              0,
            );
            const chatMessage = this.createChatMessage(message, msg);
            this.events.bitsbadgetier({
              channel,
              message: chatMessage,
              sentByCurrentUser: false,
              threshold,
              timestamp: Date.now(),
              type: TMIChatEventType.BitsBadgeTier,
            });
          } else if (msgid === 'celebrationpurchase') {
            const intensity = Utils.decodeTag(
              message.tags['msg-param-intensity'],
            );
            const effect = Utils.decodeTag(message.tags['msg-param-effect']);
            this.events.celebrationpurchase({
              channel,
              effect,
              intensity,
              user: this.createUser(message),
            });
          } else if (msgid === 'contributechannelchallenge') {
            const bits = Utils.parseInt(message.tags['msg-param-bits'], 0);
            const title = message.tags['msg-param-title'];
            const userID = message.tags['msg-param-userID'];
            const chatMessage = this.createChatMessage(message, msg);
            if (userID && title) {
              this.events.contributechannelchallenge({
                bits,
                channel,
                message: chatMessage,
                sentByCurrentUser: false,
                timestamp: Date.now(),
                title,
                type: TMIChatEventType.ContributeChannelChallenge,
                userID,
              });
            }
          } else if (msgid === 'useranniversary') {
            const years = Utils.parseInt(message.tags['msg-param-years'], 0);
            const chatMessage = this.createChatMessage(message, msg);
            this.events.useranniversary({
              channel,
              message: chatMessage,
              years,
            });
          } else if (msgid === 'communityintroduction') {
            const chatMessage = this.createChatMessage(message, msg);
            this.events.communityintroduction({
              channel,
              message: chatMessage,
            });
          } else {
            const chatMessage = this.createChatMessage(message, msg);
            this.events.usernotice(
              this.createChatMessageEvent(
                TMIChatEventType.UserNotice,
                channel,
                chatMessage,
                false,
              ),
            );
          }
          break;

        // Channel is now hosting another channel or exited host mode.
        case 'HOSTTARGET': {
          // Stopped hosting.
          const viewers = Number(msg.split(' ')[1]) || 0;

          if (msg.split(' ')[0] === '-') {
            this.commands.unhost.signal({ channel, msgid, succeeded: true });
            this.events.unhost({ channel, viewers });
          } else {
            const target = msg.split(' ')[0];
            this.commands.host.signal({ channel, msgid, succeeded: true });
            this.events.hosting({ channel, target, viewers });
          }
          break;
        }

        // Someone has been timed out or chat has been cleared by a moderator.
        case 'CLEARCHAT':
          // User has been banned / timed out by a moderator.
          if (message.params.length > 1) {
            // Duration returns null if it's a ban, otherwise it's a timeout.
            const duration = Utils.parseInt(message.tags['ban-duration'], null);
            // Escaping values: http://ircv3.net/specs/core/message-tags-3.2.html#escaping-values
            const reason = Utils.decodeTag(message.tags['ban-reason']);

            if (duration === null) {
              this.events.ban({
                channel,
                duration: null,
                reason,
                userLogin: msg,
              });
            } else {
              this.events.timeout({
                channel,
                duration,
                reason,
                userLogin: msg,
              });
            }
          } else {
            this.events.clearchat({ channel });
            this.commands.clearChat.signal({ channel, msgid, succeeded: true });
          }
          break;

        // A message has been deleted by a moderator.
        case 'CLEARMSG': {
          const targetMessageID = Utils.decodeTag(
            message.tags['target-msg-id'],
          );
          const userLogin = Utils.decodeTag(message.tags.login);
          this.events.clearmsg({
            body: msg,
            channel,
            targetMessageID,
            userLogin,
          });
          break;
        }

        // Received a reconnection request from the server.
        case 'RECONNECT':
          this.connection.onReconnect();
          break;

        // Wrong cluster.
        case 'SERVERCHANGE':
          //
          break;

        // Received when joining a channel and every time you send a PRIVMSG to a channel.
        case 'USERSTATE': {
          message.tags.username = this.session.username;
          const userstate = this.createUser(message);
          this.commands.sendMessage.signal({ channel, msgid, succeeded: true });

          // Add the client to the moderators of this room.
          if (message.tags['user-type'] === 'mod') {
            this.session.addChannelModerator(channel, this.session.username);
          }

          // Logged in and username doesn't start with justinfan.
          if (
            !Utils.isJustinfan(this.session.username) &&
            !this.session.hasJoinedChannel(channel)
          ) {
            this.session.onJoinedChannel(channel, userstate);
            this.events.joined({
              channel,
              gotUsername: true,
              username: this.session.username,
            });
          }

          // Badge have changed, update them
          if (
            message.badges &&
            this.session.updateBadges(
              channel,
              message.badges,
              message.badgeDynamicData,
            )
          ) {
            this.events.badgesupdated({
              badgeDynamicData: message.badgeDynamicData,
              badges: message.badges,
              username: Utils.username(this.session.username),
            });
          }

          // notify listener we've received a timing ack in the form of a user state with a nonce
          const nonce = Utils.decodeTag(message.tags['client-nonce']);
          if (nonce !== '') {
            this.events.timingack({ channel, nonce });
          }

          this.session.updateUserState(channel, userstate);
          break;
        }

        // Describe non-channel-specific state informations.
        case 'GLOBALUSERSTATE':
          this.logger.debug('Updated global user state', {
            tags: message.tags,
          });
          this.session.globaluserstate = message.tags;
          break;

        // Received when joining a channel and every time one of the chat room settings, like slow mode, change.
        // The message on join contains all room settings.
        case 'ROOMSTATE': {
          const parsedChannel = Utils.channel(message.params[0]);
          const channelState = this.session.getChannelState(parsedChannel);

          // Bail out if something went wrong while trying to read the persisted channel state.
          if (!channelState) {
            this.logger.warn(
              `Failed to read existing channel state for message:\n${JSON.stringify(
                message,
                null,
                4,
              )}`,
            );
            break;
          }

          const prevRoomState = channelState.roomState;
          const nextRoomState = updateRoomStateWithTags(
            prevRoomState,
            message.tags,
          );
          channelState.updateRoomState(nextRoomState);

          // We use this notice to know if we successfully joined a channel.
          if (
            this.commands.join.currentRequest &&
            this.commands.join.currentRequest.joinChannel === parsedChannel
          ) {
            this.commands.join.signal({ channel, msgid, succeeded: true });
          }

          this.events.roomstate({ channel, state: nextRoomState });

          // Handle slow mode here instead of the slow_on/off notice.
          // This room is now in slow mode. You may send messages every slow_duration seconds.
          if (
            nextRoomState.slowMode !== prevRoomState.slowMode ||
            nextRoomState.slowModeDuration !== prevRoomState.slowModeDuration
          ) {
            if (nextRoomState.slowMode) {
              this.events.slowmode({
                channel,
                enabled: true,
                length: nextRoomState.slowModeDuration || 0,
              });
            } else {
              this.events.slowmode({ channel, enabled: false, length: 0 });
            }
          }

          // Handle followers only mode here instead of the followers_on/off notice.
          // This room is now in follower-only mode.
          // This room is now in <duration> followers-only mode.
          // This room is no longer in followers-only mode.
          // duration is in minutes (number between 0 and 129600, inclusive)
          // -1 when /followersoff (number)
          if (
            nextRoomState.followersOnly !== prevRoomState.followersOnly ||
            nextRoomState.followersOnlyRequirement !==
              prevRoomState.followersOnlyRequirement
          ) {
            if (nextRoomState.followersOnly) {
              this.events.followersonly({
                channel,
                enabled: true,
                length: nextRoomState.followersOnlyRequirement || 0,
              });
            } else {
              this.events.followersonly({ channel, enabled: false, length: 0 });
            }
          }
          break;
        }

        default:
          this.logger.warn(
            `Could not parse message from tmi.twitch.tv:\n${JSON.stringify(
              message,
              null,
              4,
            )}`,
          );
          break;
      }
    } else if (message.prefix === 'jtv') {
      // Messages from jtv.
      switch (message.command) {
        case 'MODE': {
          const modUser = message.params[2];
          if (msg === '+o') {
            // Add username to the moderators.
            this.session.addChannelModerator(channel, modUser);
            this.events.mod({ channel, username: modUser });
          } else if (msg === '-o') {
            // Remove username from the moderators.
            this.session.removeChannelModerator(channel, modUser);
            this.events.unmod({ channel, username: modUser });
          }
          break;
        }

        default:
          this.logger.warn(
            `Could not parse message from jtv:\n${JSON.stringify(
              message,
              null,
              4,
            )}`,
          );
          break;
      }
    } else {
      switch (message.command) {
        case '353':
          this.events.names({
            channel: message.params[2],
            names: message.params[3].split(' '),
          });
          break;

        case '366':
          break;

        // Someone has joined the channel.
        case 'JOIN': {
          const joinedUsername = message.prefix.split('!')[0];

          // Joined a channel as a justinfan (anonymous) user.
          if (
            Utils.isJustinfan(this.session.username) &&
            this.session.username === joinedUsername
          ) {
            this.session.onJoinedChannel(channel, this.createUser(message));
            this.events.joined({
              channel,
              gotUsername: true,
              username: joinedUsername,
            });
          }

          // Someone else joined the channel, just emit the join event.
          if (this.session.username !== joinedUsername) {
            this.events.joined({
              channel,
              gotUsername: false,
              username: joinedUsername,
            });
          }

          break;
        }

        // Someone has left the channel.
        case 'PART': {
          const partedUsername = message.prefix.split('!')[0];
          const isSelf = this.session.username === partedUsername;

          // Client a channel.
          if (isSelf) {
            this.session.onPartedChannel(channel);
            this.commands.part.signal({ channel, msgid, succeeded: true });
          }

          this.events.parted({ channel, isSelf, username: partedUsername });
          break;
        }

        // Received a whisper.
        case 'WHISPER':
          // Update the tags to provide the username.
          if (!Object.prototype.hasOwnProperty.call(message.tags, 'username')) {
            message.tags.username = message.prefix.split('!')[0];
          }
          this.events.whisper({
            body: msg,
            sender: this.createUser(message),
            sentByCurrentUser: false,
          });
          break;

        case 'PRIVMSG':
          // Add username (lowercase) to the tags.
          message.tags.username = message.prefix.split('!')[0];

          // Message from JTV.
          if (message.tags.username === 'jtv') {
            if (msg.includes('hosting you')) {
              const match = NUM_VIEWERS_REGEX.exec(msg);
              let viewers = 0;
              if (match) {
                viewers = Utils.parseInt(match[1], 0);
              }

              this.events.hosted({
                channel,
                from: Utils.username(msg.split(' ')[0]),
                isAuto: msg.includes('auto'),
                viewers,
              });
            }
          } else {
            const chatMessage = this.createChatMessage(message, msg);

            // Message is an action (/me <message>).
            const actionMatch = ACTION_MESSAGE_REGEX.exec(msg);
            if (actionMatch && actionMatch.length >= 2) {
              message.tags['message-type'] = 'action';
              const actionName = actionMatch[1];
              this.events.action({
                action: actionName,
                channel,
                message: chatMessage,
                sentByCurrentUser: false,
                timestamp: Date.now(),
                type: TMIChatEventType.Action,
              });
            } else {
              let rewardID = message.tags['custom-reward-id'];
              if (msgid === 'highlighted-message') {
                rewardID = `${message.tags['room-id']}:SEND_HIGHLIGHTED_MESSAGE`;
              } else if (msgid === 'skip-subs-mode-message') {
                rewardID = `${message.tags['room-id']}:SINGLE_MESSAGE_BYPASS_SUB_MODE`;
              }

              if (rewardID) {
                this.events.channelpointsreward({
                  channel,
                  message: chatMessage,
                  rewardID,
                });
                break;
              }

              this.events.chat(
                this.createChatMessageEvent(
                  TMIChatEventType.Message,
                  channel,
                  chatMessage,
                  false,
                ),
              );
            }
          }
          break;

        default:
          this.logger.warn(
            `Could not parse message:\n${JSON.stringify(message, null, 4)}`,
          );
          break;
      }
    }
  }
}
