import type { TMIFlag, TMIFlagCategories } from './tmi-flag';
import { TMIFlagCategory } from './tmi-flag';

/**
 * Parses a message's `flags` tag. This tag is present when AutoMod detects risky content.
 * The tag string is a comma-separated list with the following format:
 * {startOffset}-{endOffset}:{category}.{severity}/{category}.{severity}...
 *
 * For example: 0-3:P.6,8-15:A.6/P.6
 * This would indicate that characters 0 through 3 are profane, and characters 8 through 15 are both
 * aggressive and profane.
 */
export function parseFlagsTag(tag?: string): Array<TMIFlag> | undefined {
  if (!tag) {
    return;
  }

  const flags: Array<TMIFlag> = [];
  tag.split(',').forEach((flagString) => {
    const [offsets, severities] = flagString.split(':');

    // Determine which categories this segment violates, filtering out those that aren't for known
    // categories, and those that aren't of severity 5 or higher.
    let matchingFlagFound = false;
    const categories = severities.split('/').reduce((obj, severityString) => {
      const [categoryString, severity] = severityString.split('.');
      const category = getFlagCategory(categoryString);
      if (parseInt(severity, 10) > 4 && category) {
        obj[category] = true;
        matchingFlagFound = true;
      }
      return obj;
    }, {} as TMIFlagCategories);

    // Only bother pushing this flag if at least one category was violated.
    if (matchingFlagFound) {
      const [start, end] = offsets.split('-');
      flags.push({
        categories,
        endIndex: parseInt(end, 10),
        startIndex: parseInt(start, 10),
      });
    }
  });

  if (flags.length > 0) {
    return flags;
  }
}

/**
 * Introduces type safety by converting a string into its corresponding TMIFlagCategory.
 * This helps us avoid type assertions while also future-proofing us against new AutoMod categories.
 */
function getFlagCategory(categoryString: string): TMIFlagCategory | null {
  switch (categoryString) {
    case TMIFlagCategory.Aggressive:
    case TMIFlagCategory.Identity:
    case TMIFlagCategory.Profanity:
    case TMIFlagCategory.Sexual:
      return categoryString;
    default:
      return null;
  }
}
