import { Utils } from '../utils';
import { TMIChannel } from './tmi-channel';
import type { TMIEmoteMap } from './tmi-emote-set';
import type { TMILogger } from './tmi-logger';
import type { TMIMessageTags } from './tmi-message';
import type { TMIRoomState } from './tmi-room-state';
import type { TMIUser } from './tmi-user';

export class TMISession {
  public emoteMap: TMIEmoteMap = {};
  public globaluserstate: TMIMessageTags = {};
  public lastChannelJoined = '';
  public username = '';

  private readonly channelstate: Record<string, TMIChannel> = {};
  private readonly logger: TMILogger;

  constructor(logger: TMILogger) {
    this.logger = logger;
  }

  public reset(): void {
    const channels = Object.keys(this.channelstate);
    for (const channel of channels) {
      delete this.channelstate[channel];
    }
  }

  public hasJoinedChannel(channel: string): boolean {
    channel = Utils.channel(channel);
    return this.channelstate[channel] !== undefined;
  }

  public onJoinedChannel(channel: string, userState: TMIUser): void {
    const userID = this.globaluserstate['user-id'];
    if (userID && userState.userID === '') {
      userState.userID = userID;
    }

    channel = Utils.channel(channel);
    const channelState = new TMIChannel(userState);
    this.lastChannelJoined = channel;
    this.channelstate[channel] = channelState;
    this.logger.debug('[Session] Joined channel', {
      channel,
      newChannelState: this.channelstate,
    });
  }

  public onPartedChannel(channel: string): void {
    channel = Utils.channel(channel);
    if (this.channelstate[channel]) {
      delete this.channelstate[channel];
      const remainingChannels = Object.keys(this.channelstate);
      if (this.lastChannelJoined === channel && remainingChannels.length > 0) {
        this.lastChannelJoined = remainingChannels[0];
      }
      this.logger.debug('[Session] Parted channel', {
        channel,
        newChannelState: this.channelstate,
      });
    }
  }

  public addChannelModerator(channel: string, username: string): void {
    channel = Utils.channel(channel);
    username = Utils.username(username);
    const channelState = this.getChannelState(channel);
    if (!channelState) {
      return;
    }
    channelState.addModerator(username);
  }

  public removeChannelModerator(channel: string, username: string): void {
    channel = Utils.channel(channel);
    username = Utils.username(username);
    const channelState = this.getChannelState(channel);
    if (!channelState) {
      return;
    }
    channelState.removeModerator(username);
  }

  public getRoomState(channel?: string): TMIRoomState | null {
    channel = Utils.channel(channel);
    const channelState = this.getChannelState(channel);
    if (!channelState) {
      return null;
    }

    return channelState.roomState;
  }

  public getUserState(channel?: string): TMIUser | null {
    channel = Utils.channel(channel);
    const channelState = this.getChannelState(channel);
    if (!channelState) {
      return null;
    }

    return channelState.userState;
  }

  public updateUserState(channel: string, userState: TMIUser): void {
    channel = Utils.channel(channel);
    const channelState = this.getChannelState(channel);
    if (!channelState) {
      return;
    }
    channelState.updateUserState(userState);
    this.logger.debug('Updated user state', userState);
  }

  public updateEmoteMap(emoteMap: TMIEmoteMap): void {
    this.emoteMap = emoteMap;
  }

  public updateBadges(
    channel: string,
    badges: Record<string, string>,
    dynamicData?: Record<string, string>,
  ): boolean {
    dynamicData = dynamicData || {};
    channel = Utils.channel(channel);
    const channelState = this.getChannelState(channel);
    if (!channelState) {
      return false;
    }

    return channelState.updateBadges(badges, dynamicData);
  }

  public getChannelState(channel?: string): TMIChannel | null {
    if (!channel) {
      this.logger.warn(
        'Unable to get channel. Channel name is empty or undefined.',
      );
      return null;
    }
    const channelState = this.channelstate[channel];
    if (!channelState) {
      this.logger.warn(
        'Unable to get channel. Channel state is missing from session.',
        { channel },
      );
      return null;
    }

    return channelState;
  }
}
