import { TMIConnectionState } from '../../models';
import type { TMIConnection } from '../../tmi-connection';
import { mockTMICommands } from './tmi-commands';
import { mockTMILogger } from './tmi-logger';

let latestTMIConnection: MockTMIConnection;
// defined outside the class to persist between instantiations of TMIConnection
let mockConnectionState: TMIConnectionState | null;

export class MockTMIConnection {
  public logger = mockTMILogger();
  public getCommandTimeout = jest.fn();
  public send = jest.fn();
  public injectMessage = jest.fn();
  public commands = mockTMICommands();
  public disconnect = jest.fn();
  public notifyReconnect = jest.fn();
  public notifyReconnected = jest.fn();
  public suppressEvents = jest.fn();
  public unsuppressEvents = jest.fn();
  public maxReconnectAttempts = 1;

  constructor() {
    latestTMIConnection = this;
  }

  public isConnected = (): boolean =>
    mockConnectionState === TMIConnectionState.Connected;

  // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
  public tryConnect = async () =>
    Promise.resolve({ state: mockConnectionState });
}

export function setTMIConnectionState(state: TMIConnectionState): void {
  mockConnectionState = state;
}

/**
 * Returns a mocked TMIConnection class.
 *
 * Usage:
 *  const MockTMIConnection = mockTMIConnection();
 *  const connection: TMIConnection = new MockTMIConnection();
 */
export function mockTMIConnection(): TMIConnection {
  return {
    commands: mockTMICommands(),
    disconnect: jest.fn(),
    getCommandTimeout: jest.fn().mockReturnValue(1000),
    logger: mockTMILogger(),
    onReconnect: jest.fn(),
    pong: jest.fn(),
    send: jest.fn(),
  } as any;
}

export function getLatestTMIConnection(): MockTMIConnection {
  return latestTMIConnection;
}

export function resetLatestTMIConnection(): void {
  mockConnectionState = null;
}
