import type { EventProcessors } from '../../event-processors';
import type {
  TMISubscriptionFunFact,
  TMISubscriptionGoalData,
  TMIUser,
} from '../../models';
import { TMIChatEventType } from '../../models';
import type { TMIConnection } from '../../tmi-connection';

export const mockLoggedInUser: TMIUser = {
  badgeDynamicData: null,
  badges: null,
  bits: 0,
  color: 'red',
  displayName: 'shrood',
  emotes: {},
  id: '1234',
  turbo: false,
  userID: '',
  userType: '',
  username: 'shrood',
};

export const mockAnonymousUser: TMIUser = {
  badgeDynamicData: null,
  badges: null,
  bits: 0,
  color: 'red',
  displayName: 'justinfan1234',
  emotes: {},
  id: '',
  turbo: false,
  userID: '',
  userType: '',
  username: 'justinfan1234',
};

export const mockSubscriptionFunFactFollowingAge: TMISubscriptionFunFact = {
  followingAge: 120,
  funFactType: 'following-age',
};

export const mockSubscriptionFunFactMostUsedEmote: TMISubscriptionFunFact = {
  funFactType: 'emote-usage',
  mostUsedEmoteID: '12345',
};

export function createSubscriptionFunFactFollowingAgeTags(): Array<string> {
  return [
    `msg-param-fun-fact-type=following-age`,
    `msg-param-fun-fact-content=120`,
  ];
}

export function createSubscriptionFunFactMostUsedEmoteTags(): Array<string> {
  return [
    `msg-param-fun-fact-type=emote-usage`,
    `msg-param-fun-fact-content=12345`,
  ];
}

export const mockGoalData: TMISubscriptionGoalData = {
  contributionType: 'SUB_POINTS',
  currentContributions: 87,
  description: 'hello there',
  targetContributions: 100,
  userContributions: 2,
};

export function createGoalTags(
  mockGoal: TMISubscriptionGoalData,
): Array<string> {
  return [
    `msg-param-goal-description=${mockGoal.description?.replace(' ', '\\s')}`,
    `msg-param-goal-contribution-type=${mockGoal.contributionType}`,
    `msg-param-goal-current-contributions=${mockGoal.currentContributions}`,
    `msg-param-goal-target-contributions=${mockGoal.targetContributions}`,
    `msg-param-goal-user-contributions=${mockGoal.userContributions}`,
  ];
}

/** Takes a TMIUser and returns an array of message tags of their user info */
export function createUserTags(mockUser: TMIUser): Array<string> {
  return [
    `username=${mockUser.username}`,
    `color=${mockUser.color}`,
    `displayName=${mockUser.displayName}`,
    `id=${mockUser.id}`,
    `userID=${mockUser.userID}`,
  ];
}

export interface CreateMessageOptions {
  channel?: string | null;
  command?: string;
  msgid?: string;
  params?: string[];
  prefix?: string;
  tags?: string[];
}

export interface CreateMessageResult {
  message: string;
  /** Individual parts of the message, used for assertions */
  messageParts: {
    /**
     * Should be undefined if the message should have the "default" mock channel
     * Should be null if the message should not have a channel at all
     */
    channel?: string | null;
    command?: string;
    params?: string;
    prefix?: string;
  };
}

/**
 * Creates a raw TMI message string as well as the individual parts to assert
 * against.
 */
export function createMessage(
  options: CreateMessageOptions = {},
): CreateMessageResult {
  const tags: string[] = options.tags || [];
  const prefix = options.prefix || '';
  const command = options.command || 'PRIVMSG';
  const params: string[] = options.params || [];
  const channel =
    options.channel !== undefined ? options.channel : '#testchannel';

  if (options.msgid) {
    tags.push(`msg-id=${options.msgid}`);
  }

  const rawMessage = [
    // Tags
    `${tags.length ? '@' : ''}${tags.join(';')}`,

    // Prefix
    `:${prefix}`,

    // Command
    command,

    // Channel
    channel,

    // Params
    params.join(' '),
  ].join(' ');

  return {
    message: rawMessage,
    messageParts: {
      channel,
      command,
      prefix,
    },
  };
}

/** Creates a TMI prefix with a given username. */
export function createPrefixWithUsername(username: string): string {
  return `${username}!${username}@${username}.tmi.twitch.tv`;
}

export function expectNoticeEventRaised(
  events: EventProcessors,
  options: { body: string; channel?: string | null | undefined; msgid: string },
): void {
  expect(events.notice).toHaveBeenLastCalledWith({
    body: options.body,
    channel: options.channel,
    msgid: options.msgid,
    timestamp: Date.now(),
    type: TMIChatEventType.Notice,
  });
}

/**
 * Checks the events, connetion and command that no relevant methods
 * were invoked.
 */
export function expectNoop(
  events: EventProcessors,
  connection: TMIConnection,
): void {
  // EventProcessor
  const eventProcessorMethods: Array<keyof EventProcessors> = [
    'unsuppress',
    'suppress',
    'action',
    'disconnected',
    'connecting',
    'connected',
    'reconnecting',
    'reconnected',
    'chat',
    'joined',
    'parted',
    'followersonly',
    'badgesupdated',
    'clearchat',
    'hosting',
    'hosted',
    'ban',
    'timeout',
    'emoteonlymode',
    'mods',
    'names',
    'notice',
    'usernotice',
    'purchase',
    'crate',
    'rewardgift',
    'raid',
    'ritual',
    'r9kmode',
    'resub',
    'extendsub',
    'roomstate',
    'slowmode',
    'subgift',
    'submysterygift',
    'subscribers',
    'charity',
    'subscription',
    'unhost',
    'unraid',
    'mod',
    'unmod',
    'whisper',
    'primepaidupgrade',
    'communitypayforward',
    'standardpayforward',
    'celebrationpurchase',
  ];

  for (const method of eventProcessorMethods) {
    expect(events[method]).not.toHaveBeenCalled();
  }

  // TMIConnection
  const connectionMethods: Array<keyof TMIConnection> = [
    'pong',
    'disconnect',
    'onReconnect',
    'send',
  ];

  for (const method of connectionMethods) {
    expect(connection[method]).not.toHaveBeenCalled();
  }

  // As fun as it would be to use conditional types to type this as an array
  // of properties of TMICommands that implement TMICommand, we need a newer
  // TS version to do that.
  const commands = [
    connection.commands.color,
    connection.commands.commercial,
    connection.commands.host,
    connection.commands.followersOnlyOn,
    connection.commands.followersOnlyOff,
    connection.commands.emoteOnlyModeOn,
    connection.commands.emoteOnlyModeOff,
    connection.commands.subscriberModeOn,
    connection.commands.subscriberModeOff,
    connection.commands.rk9ModeOn,
    connection.commands.rk9ModeOff,
    connection.commands.connect,
    connection.commands.ping,
    connection.commands.ban,
    connection.commands.unban,
    connection.commands.timeout,
    connection.commands.slowModeOn,
    connection.commands.slowModeOff,
    connection.commands.unhost,
    connection.commands.whisper,
    connection.commands.clearChat,
    connection.commands.part,
    connection.commands.sendMessage,
    connection.commands.join,
  ];

  for (const command of commands) {
    expect(command.signal).not.toHaveBeenCalled();
    expect(command.execute).not.toHaveBeenCalled();
  }
}
