import type { ClientConfiguration } from './client-configuration';
import type { TMIIdentityOptions, TMISession } from './models';
import { TMIConnectionState } from './models';
import {
  MockTMIConnection,
  getLatestTMIConnection,
  mockTMILogger,
  resetLatestTMIConnection,
  setTMIConnectionState,
} from './tests';
import { TMIClient } from './tmi-client';

let mockClientConfiguration: ClientConfiguration;
jest.mock('./client-configuration', () => {
  const actual = jest.requireActual('./client-configuration');
  return {
    ClientConfiguration: () => {
      mockClientConfiguration = new actual.ClientConfiguration({
        connection: {
          port: 443,
          secure: true,
          server: 'irc-ws.chat.twitch.tv',
        },
        logger: console,
      });

      return mockClientConfiguration;
    },
  };
});

let mockTMISession: TMISession;
jest.mock('./models', () => {
  return {
    ...jest.requireActual('./models'),
    TMISession: () => {
      mockTMISession = {
        updateBadges: jest.fn(),
        updateEmoteMap: jest.fn(),
      } as any;

      return mockTMISession;
    },
  };
});

jest.mock('./tmi-connection', () => {
  return {
    TMIConnection: () => new MockTMIConnection(),
  };
});

function setup() {
  const client = new TMIClient({
    connection: {
      port: 443,
      secure: true,
      server: 'irc-ws.chat.twitch.tv',
    },
    logger: mockTMILogger(),
  });

  return {
    client,
    configuration: mockClientConfiguration,
    connection: getLatestTMIConnection(),
    session: mockTMISession,
  };
}

describe('TMI Client', () => {
  afterEach(() => {
    resetLatestTMIConnection();
  });

  it('disconnects', () => {
    const { client } = setup();
    client.disconnect();
    expect(getLatestTMIConnection().disconnect).toHaveBeenCalledWith(false);
  });

  it('connects', async () => {
    const { client, connection } = setup();
    setTMIConnectionState(TMIConnectionState.Connected);
    const res = await client.connect();

    expect(res.state).toBe(TMIConnectionState.Connected);
    expect(connection.disconnect).not.toHaveBeenCalled();
  });

  it('attempts to reconnect on connect failure', async () => {
    const { client } = setup();
    setTMIConnectionState(TMIConnectionState.Disconnected);
    client.reconnect = jest.fn();
    const res = await client.connect();
    expect(client.reconnect).toHaveBeenCalled();
    expect(res.state).toBe(TMIConnectionState.Reconnecting);
  });

  it('call TMIConnection disconnect with true when triggerArtificialDisconnect called and isConnected', () => {
    const { client, connection } = setup();
    setTMIConnectionState(TMIConnectionState.Connected);
    client.triggerArtificialDisconnect();
    expect(connection.disconnect).toHaveBeenCalledWith(true);
  });

  it('does not call TMIConnection disconnect with true when triggerArtificialDisconnect called and not isConnected', () => {
    const { client, connection } = setup();
    setTMIConnectionState(TMIConnectionState.Disconnected);
    client.triggerArtificialDisconnect();
    expect(connection.disconnect).not.toHaveBeenCalled();
  });

  it('passes the correct data to injectMessage', () => {
    const { client, connection } = setup();
    client.injectMessage('test string');
    expect(connection.injectMessage).toHaveBeenCalledWith('test string');
  });

  it('calls configuration to update identity', () => {
    const { client, configuration } = setup();
    const mockIdentity: TMIIdentityOptions = {
      authToken: '5678',
      username: 'kappa',
    };

    configuration.updateIdentity = jest.fn();
    client.updateIdentity(mockIdentity);

    expect(configuration.updateIdentity).toHaveBeenCalledWith(mockIdentity);
  });

  it('updates emote map', () => {
    const { client, session } = setup();
    client.updateEmoteMap([
      { emotes: [{ id: '1', token: ':-?[z|Z|\\|]' }], id: '1' },
    ]);
    expect(session.updateEmoteMap).toHaveBeenCalledWith({
      ':-?[z|Z|\\|]': { id: '1', token: ':-?[z|Z|\\|]' },
    });
  });

  it('updates channel badges', () => {
    const { client, session } = setup();
    const badges = {
      badge1: '1',
      badge2: '3',
    };
    const dynamicData = {};

    client.updateChannelBadges('qa_test_01', badges, dynamicData);
    expect(session.updateBadges).toHaveBeenCalledWith(
      'qa_test_01',
      badges,
      dynamicData,
    );
  });

  it('sends commands', async () => {
    const { client } = setup();
    client.commands.processCommand = jest.fn();
    await client.sendCommand('qa_test_01', '/me jokes', {});
    expect(client.commands.processCommand).toHaveBeenCalledWith(
      'qa_test_01',
      '/me jokes',
      {},
    );
  });

  it('executes the join channel command', () => {
    const { client } = setup();
    client.joinChannel('qa_test_01');
    expect(client.commands.join.execute).toHaveBeenCalledWith({
      joinChannel: 'qa_test_01',
    });
  });

  it('executes the part channel command', () => {
    const { client } = setup();
    client.partChannel('qa_test_01');
    expect(client.commands.part.execute).toHaveBeenCalledWith({
      partChannel: 'qa_test_01',
    });
  });

  it('returns true when is connected', () => {
    const { client } = setup();
    setTMIConnectionState(TMIConnectionState.Connected);
    expect(client.isConnected()).toBeTruthy();
  });

  it('returns false when is disconnected', () => {
    const { client } = setup();
    setTMIConnectionState(TMIConnectionState.Disconnected);
    expect(client.isConnected()).toBeFalsy();
  });

  describe('reconnect', () => {
    // workaround for timer functionality in Jest 27, pending resolution
    // tmi-client is uniquely pathologic to jest/sinon's timers
    // https://github.com/facebook/jest/issues/2157
    async function runAllTimersThroughPromises(): Promise<void> {
      jest.runAllTimers();
      const reconnectPromiseDepth = 7;
      for (let i = 0; i < reconnectPromiseDepth; i++) {
        await Promise.resolve();
      }
    }

    it('calls onReconnectSuccess on reconnect success', async () => {
      const { client, connection } = setup();
      setTMIConnectionState(TMIConnectionState.Connected);
      client.reconnect();
      await runAllTimersThroughPromises();

      const newConnection = getLatestTMIConnection();
      expect(connection.notifyReconnect).toHaveBeenCalled();
      expect(connection.suppressEvents).toHaveBeenCalled();
      expect(newConnection.unsuppressEvents).toHaveBeenCalled();
      expect(connection.disconnect).toHaveBeenCalled();
      expect(newConnection.notifyReconnected).toHaveBeenCalledWith(1, 'None');
    });

    it('does not call onReconnectSuccess on reconnect failure', async () => {
      const { client, connection } = setup();
      setTMIConnectionState(TMIConnectionState.Disconnected);
      client.reconnect();
      await runAllTimersThroughPromises();

      expect(connection.notifyReconnect).toHaveBeenCalled();
      expect(connection.notifyReconnected).not.toHaveBeenCalled();
    });
  });
});
