import EventEmitter from 'eventemitter3';
import { ClientConfiguration } from './client-configuration';
import type {
  TMICommandResponse,
  TMICommandResult,
  TMIJoinRequest,
  TMIPartRequest,
} from './commands';
import type {
  TMIClientOptions,
  TMIConnectionResult,
  TMIEmoteSet,
  TMIEventName,
  TMIIdentityOptions,
  TMILogger,
} from './models';
import { TMIConnectionState, TMISession } from './models';
import type { TMIAdditionalMetadata } from './tmi-command-processor';
import type { TMICommands } from './tmi-commands';
import { TMIConnection } from './tmi-connection';
import { TMIEventHandlers } from './tmi-events-handlers';
import { TMIParser } from './tmi-parser';
import { Utils, createEmoteMap } from './utils';

export class TMIClient {
  public commands: TMICommands;
  public readonly events: TMIEventHandlers;
  public readonly logger: TMILogger;
  private readonly configuration: ClientConfiguration;
  private readonly eventEmitter: EventEmitter<TMIEventName>;
  private readonly session: TMISession;
  private readonly parser: TMIParser;

  private connection: TMIConnection;

  // Reconnect state management
  private isReconnecting = false;
  private reconnectAttempts = 0;

  constructor(opts: TMIClientOptions) {
    this.logger = opts.logger || console;
    this.eventEmitter = new EventEmitter();
    this.session = new TMISession(this.logger);
    this.events = new TMIEventHandlers(this.eventEmitter, this.logger);
    this.parser = new TMIParser(this.logger);
    this.configuration = new ClientConfiguration(opts);
    this.connection = new TMIConnection(
      this,
      this.configuration,
      false,
      this.eventEmitter,
      this.session,
      this.parser,
      this.logger,
    );
    this.commands = this.connection.commands;

    this.logger.debug('Created', {
      pingInterval: this.configuration.pingInterval,
      port: this.configuration.port,
      reconnectJitter: this.configuration.reconnectJitter,
      server: this.configuration.server,
    });
  }

  public disconnect(): void {
    if (this.connection) {
      this.connection.disconnect(false);
    }
  }

  public async connect(): Promise<TMIConnectionResult> {
    try {
      const res = await this.connection.tryConnect();
      if (res.state === TMIConnectionState.Connected) {
        return res;
      }
    } catch (err) {
      this.connection.disconnect(false);
      this.logger.warn(
        'Failed initial connect attempt. Starting reconnect process...',
      );
    }

    // Intentionally not awaited
    this.reconnect();

    return { state: TMIConnectionState.Reconnecting };
  }

  public triggerArtificialDisconnect(): void {
    if (!this.isConnected()) {
      return;
    }
    this.connection.disconnect(true);
  }

  public injectMessage(data: string): void {
    this.connection.injectMessage(data);
  }

  public updateExperimentFlags(skipAutoRejoin: boolean): void {
    this.configuration.updateRejoinConfiguration(skipAutoRejoin);
  }

  public updateIdentity(identity: TMIIdentityOptions): void {
    this.configuration.updateIdentity(identity);
  }

  public updateServerHost(host: string): void {
    this.configuration.updateServerHost(host);
  }

  public updateEmoteMap(emoteSets: TMIEmoteSet[]): void {
    const emoteMap = createEmoteMap(emoteSets);
    this.session.updateEmoteMap(emoteMap);
    this.logger.debug('Updated emote map', { emoteMap });
  }

  public updateChannelBadges(
    channel: string,
    badges: Record<string, string>,
    dynamicData: Record<string, string>,
  ): void {
    this.session.updateBadges(channel, badges, dynamicData);
    this.logger.debug(
      'Updated channel badges and dynamic data',
      { badges },
      { dynamicData },
    );
  }

  public async sendCommand(
    channel: string,
    message: string,
    additionalMetadata: TMIAdditionalMetadata | undefined = {},
  ): Promise<any> {
    try {
      await this.commands.processCommand(channel, message, additionalMetadata);
    } catch (err) {
      // QOS: Failed to send message.
      this.logger.trackAndWarn(
        'chat_message_sending_failure',
        {
          reason: Utils.isError(err)
            ? err.toString()
            : 'chat_message_sending_failure',
        },
        'Failed to send message, or message timeout',
        err,
      );
    }
  }

  public async joinChannel(
    channel: string,
  ): Promise<TMICommandResult<TMIJoinRequest, TMICommandResponse>> {
    return this.commands.join.execute({ joinChannel: channel });
  }

  public async partChannel(
    channel: string,
  ): Promise<TMICommandResult<TMIPartRequest, TMICommandResponse>> {
    return this.commands.part.execute({ partChannel: channel });
  }

  public isConnected(): boolean | null {
    return this.connection.isConnected();
  }

  public async reconnect(): Promise<void> {
    if (this.isReconnecting) {
      return;
    }

    if (this.reconnectAttempts >= this.configuration.maxReconnectAttempts) {
      return;
    }

    ++this.reconnectAttempts;
    const reconnectDelay = this.configuration.getReconnectDelay(
      this.reconnectAttempts,
    );
    this.logger.debug(`Reconnect waiting ${reconnectDelay} ms`);
    await this.delay(reconnectDelay);

    const success = await this.tryReconnect();
    if (success) {
      this.onReconnectSuccess();
    } else {
      this.onReconnectFail();
    }
  }

  private onReconnectSuccess() {
    this.connection.notifyReconnected(this.reconnectAttempts, 'None');
    this.logger.debug('Reconnect succesful');
    this.isReconnecting = false;
    this.reconnectAttempts = 0;
  }

  private onReconnectFail() {
    this.logger.debug('Reconnect failed');
    this.isReconnecting = false;
    this.reconnect();
  }

  private async delay(ms: number) {
    return new Promise((resolve) => setTimeout(resolve, ms));
  }

  private async tryReconnect() {
    this.logger.debug('Reconnect initiated');
    try {
      this.connection.notifyReconnect('None');
      const newConnection = new TMIConnection(
        this,
        this.configuration,
        true,
        this.eventEmitter,
        this.session,
        this.parser,
        this.logger,
      );
      const result = await newConnection.tryConnect();
      if (result.state === TMIConnectionState.Connected) {
        this.logger.debug('Reconnect connection succeeded', result);
        const oldConnection = this.connection;
        oldConnection.suppressEvents();
        newConnection.unsuppressEvents();
        this.connection = newConnection;
        this.commands = newConnection.commands;
        oldConnection.disconnect(false);
        return true;
      }
      this.logger.info('Reconnect connection failed', result);
      return false;
    } catch (err) {
      this.logger.warn('Reconnect failed', err);
      return false;
    }
  }
}
