import { mockTMICommands } from './tests';
import { TMICommandProcessor } from './tmi-command-processor';

describe('TMICommandProcessor', () => {
  const setup = () => {
    const commands = mockTMICommands();
    const processor = new TMICommandProcessor(commands);
    return { commands, processor };
  };

  it('can be constructed', () => {
    const { processor } = setup();
    expect(processor).toBeTruthy();
  });

  describe('executeCommand()', () => {
    it('rejects an empty command', async () => {
      const { processor } = setup();
      await expect(
        processor.executeCommand('testChannel', '', {}),
      ).rejects.toBe('Could not execute empty command.');
    });

    it('rejects a command with an empty channel', async () => {
      const { processor } = setup();
      await expect(processor.executeCommand('', 'color', {})).rejects.toBe(
        'Could not execute command without channel.',
      );
    });

    it('sends general messages that are not commands', () => {
      const { commands, processor } = setup();
      processor.executeCommand('testChannel', 'hello', {});
      expect(commands.sendMessage.execute).toHaveBeenLastCalledWith({
        additionalMetadata: {},
        channel: 'testChannel',
        message: 'hello',
      });
    });

    it('executes "color" commands', () => {
      const { commands, processor } = setup();
      processor.executeCommand('testChannel', '/color red', {});
      expect(commands.sendMessage.execute).not.toHaveBeenCalled();
      expect(commands.color.execute).toHaveBeenLastCalledWith({
        channel: 'testChannel',
        color: 'red',
      });
    });

    it('sends other commands as regular messages', () => {
      const { commands, processor } = setup();
      processor.executeCommand('testChannel', '/host profmulo', {});
      expect(commands.sendMessage.execute).toHaveBeenLastCalledWith({
        channel: 'testChannel',
        message: '/host profmulo',
      });
    });

    it('sends forward slash commands as regular messages', () => {
      const { commands, processor } = setup();
      processor.executeCommand('testChannel', '\\wow lol', {});
      expect(commands.sendMessage.execute).toHaveBeenLastCalledWith({
        additionalMetadata: {},
        channel: 'testChannel',
        message: '\\wow lol',
      });
    });
  });
});
