import type { TMIGenericCommandResult } from './commands';
import type { Reply } from './models';
import type { TMICommands } from './tmi-commands';

export interface TMIAdditionalMetadata {
  clientNonce?: string;
  crowdChantParentMessageID?: string;
  reply?: Reply;
}

interface TMICommandArgs {
  name: string;
  params: string[];
}

export class TMICommandProcessor {
  private readonly commands: TMICommands;

  constructor(commands: TMICommands) {
    this.commands = commands;
  }

  public async executeCommand(
    channel: string,
    message: string,
    additionalMetadata: TMIAdditionalMetadata,
  ): Promise<TMIGenericCommandResult> {
    if (!message) {
      return Promise.reject('Could not execute empty command.');
    }

    if (!channel) {
      return Promise.reject('Could not execute command without channel.');
    }

    const args = this.getCommandArgs(message);
    // If not a command, assume it's a general message type
    if (!args) {
      return this.commands.sendMessage.execute({
        additionalMetadata,
        channel,
        message,
      });
    }

    // Look for special commands
    switch (args.name.toLowerCase()) {
      case 'color':
        return this.commands.color.execute({ channel, color: args.params[0] });
      default:
        return this.commands.sendMessage.execute({ channel, message });
    }
  }

  private getCommandArgs(message: string): TMICommandArgs | null {
    if (message.charAt(0) !== '/') {
      return null;
    }
    const commandParts = message.substring(1).split(' ');
    return {
      name: commandParts[0],
      params: commandParts.slice(1),
    };
  }
}
