import type { TMIGenericCommandResult } from './commands';
import {
  BanCommand,
  ClearChatCommand,
  ColorCommand,
  CommercialCommand,
  ConnectCommand,
  EmoteOnlyModeOffCommand,
  EmoteOnlyModeOnCommand,
  FollowersOnlyCommand,
  FollowersOnlyOffCommand,
  HostCommand,
  JoinCommand,
  PartCommand,
  PingCommand,
  RK9ModeOffCommand,
  RK9ModeOnCommand,
  SendMessageCommand,
  SlowModeOffCommand,
  SlowModeOnCommand,
  SubscriberModeOffCommand,
  SubscriberModeOnCommand,
  TimeoutCommand,
  UnbanCommand,
  UnhostCommand,
  WhisperCommand,
} from './commands';
import type { EventProcessors } from './event-processors';
import type { TMISession } from './models';
import type { TMIAdditionalMetadata } from './tmi-command-processor';
import { TMICommandProcessor } from './tmi-command-processor';
import type { TMIConnection } from './tmi-connection';

export class TMICommands {
  public readonly ban: BanCommand;
  public readonly clearChat: ClearChatCommand;
  public readonly color: ColorCommand;
  public readonly commercial: CommercialCommand;
  public readonly connect: ConnectCommand;
  public readonly emoteOnlyModeOff: EmoteOnlyModeOffCommand;
  public readonly emoteOnlyModeOn: EmoteOnlyModeOnCommand;
  public readonly followersOnlyOff: FollowersOnlyOffCommand;
  public readonly followersOnlyOn: FollowersOnlyCommand;
  public readonly host: HostCommand;
  public readonly join: JoinCommand;
  public readonly part: PartCommand;
  public readonly ping: PingCommand;
  public readonly rk9ModeOff: RK9ModeOffCommand;
  public readonly rk9ModeOn: RK9ModeOnCommand;
  public readonly sendMessage: SendMessageCommand;
  public readonly slowModeOff: SlowModeOffCommand;
  public readonly slowModeOn: SlowModeOnCommand;
  public readonly subscriberModeOff: SubscriberModeOffCommand;
  public readonly subscriberModeOn: SubscriberModeOnCommand;
  public readonly timeout: TimeoutCommand;
  public readonly unban: UnbanCommand;
  public readonly unhost: UnhostCommand;
  public readonly whisper: WhisperCommand;

  private readonly connection: TMIConnection;
  private readonly session: TMISession;
  private readonly events: EventProcessors;
  private readonly commandProcessor: TMICommandProcessor;

  constructor(
    connection: TMIConnection,
    session: TMISession,
    events: EventProcessors,
  ) {
    this.connection = connection;
    this.session = session;
    this.events = events;
    this.commandProcessor = new TMICommandProcessor(this);

    this.ban = new BanCommand(this.connection, this.session);
    this.clearChat = new ClearChatCommand(this.connection, this.session);
    this.color = new ColorCommand(this.connection, this.session);
    this.commercial = new CommercialCommand(this.connection, this.session);
    this.connect = new ConnectCommand(this.connection, this.session);
    this.emoteOnlyModeOff = new EmoteOnlyModeOffCommand(
      this.connection,
      this.session,
    );
    this.emoteOnlyModeOn = new EmoteOnlyModeOnCommand(
      this.connection,
      this.session,
    );
    this.followersOnlyOff = new FollowersOnlyOffCommand(
      this.connection,
      this.session,
    );
    this.followersOnlyOn = new FollowersOnlyCommand(
      this.connection,
      this.session,
    );
    this.host = new HostCommand(this.connection, this.session);
    this.join = new JoinCommand(this.connection, this.session);
    this.part = new PartCommand(this.connection, this.session);
    this.ping = new PingCommand(this.connection, this.session);
    this.rk9ModeOff = new RK9ModeOffCommand(this.connection, this.session);
    this.rk9ModeOn = new RK9ModeOnCommand(this.connection, this.session);
    this.sendMessage = new SendMessageCommand(
      this.connection,
      this.session,
      this.events,
    );
    this.slowModeOff = new SlowModeOffCommand(this.connection, this.session);
    this.slowModeOn = new SlowModeOnCommand(this.connection, this.session);
    this.subscriberModeOff = new SubscriberModeOffCommand(
      this.connection,
      this.session,
    );
    this.subscriberModeOn = new SubscriberModeOnCommand(
      this.connection,
      this.session,
    );
    this.timeout = new TimeoutCommand(this.connection, this.session);
    this.unban = new UnbanCommand(this.connection, this.session);
    this.unhost = new UnhostCommand(this.connection, this.session);
    this.whisper = new WhisperCommand(this.connection, this.session);
  }

  public async processCommand(
    channel: string,
    command: string,
    additionalMetadata: TMIAdditionalMetadata,
  ): Promise<TMIGenericCommandResult> {
    return this.commandProcessor.executeCommand(
      channel,
      command,
      additionalMetadata,
    );
  }

  public failAll(msgid: string, channel: string): void {
    this.ban.signal({ channel, msgid, succeeded: false });
    this.clearChat.signal({ channel, msgid, succeeded: false });
    this.color.signal({ channel, msgid, succeeded: false });
    this.commercial.signal({ channel, msgid, succeeded: false });
    this.emoteOnlyModeOff.signal({ channel, msgid, succeeded: false });
    this.emoteOnlyModeOn.signal({ channel, msgid, succeeded: false });
    this.followersOnlyOff.signal({ channel, msgid, succeeded: false });
    this.followersOnlyOn.signal({ channel, msgid, succeeded: false });
    this.host.signal({ channel, msgid, succeeded: false });
    this.rk9ModeOff.signal({ channel, msgid, succeeded: false });
    this.rk9ModeOn.signal({ channel, msgid, succeeded: false });
    this.sendMessage.signal({ channel, msgid, succeeded: false });
    this.slowModeOff.signal({ channel, msgid, succeeded: false });
    this.slowModeOn.signal({ channel, msgid, succeeded: false });
    this.subscriberModeOff.signal({ channel, msgid, succeeded: false });
    this.subscriberModeOn.signal({ channel, msgid, succeeded: false });
    this.timeout.signal({ channel, msgid, succeeded: false });
    this.unban.signal({ channel, msgid, succeeded: false });
    this.unhost.signal({ channel, msgid, succeeded: false });
    this.whisper.signal({ channel, msgid, succeeded: false });
  }
}
