import { TMIConnectionState } from './models';
import {
  PARSED_VALID_TMI_MESSAGE,
  VALID_TMI_MESSAGE,
  mockClientConfiguration,
  mockEventEmitter,
  mockTMIClient,
  mockTMILogger,
  mockTMIParser,
  mockTMISession,
  mockWebSocket,
} from './tests';
import { TMIConnection } from './tmi-connection';

function setup() {
  const tmiClient = mockTMIClient();
  const clientConfiguration = mockClientConfiguration();
  const tmiLogger = mockTMILogger();
  const eventEmitter = mockEventEmitter();
  const tmiSession = mockTMISession();
  const tmiParser = mockTMIParser();

  const tmiConnection = new TMIConnection(
    tmiClient,
    clientConfiguration,
    false,
    eventEmitter,
    tmiSession,
    tmiParser,
    tmiLogger,
  );

  return {
    clientConfiguration,
    eventEmitter,
    tmiClient,
    tmiConnection,
    tmiParser,
    tmiSession,
  };
}

let originalWebSocket: WebSocket;

describe('TMIConnection', () => {
  beforeEach(() => {
    originalWebSocket = global.WebSocket as any;
    (global.WebSocket as any) = mockWebSocket();
  });

  afterEach(() => {
    (global.WebSocket as any) = originalWebSocket;
  });

  it('instantiates without error', () => {
    const { tmiConnection } = setup();
    expect(tmiConnection).toBeDefined();
  });

  describe('.tryConnect()', () => {
    it('instantiates a new WebSocket with the correct connectURL', () => {
      const { clientConfiguration, tmiConnection } = setup();
      tmiConnection.tryConnect();
      expect((global.WebSocket as any).instances[0]).toHaveProperty(
        'url',
        clientConfiguration.connectUrl,
      );
    });

    it('sends a connect command', () => {
      const { tmiConnection } = setup();
      tmiConnection.commands.connect.execute = jest.fn();
      tmiConnection.tryConnect();
      expect(tmiConnection.commands.connect.execute).toHaveBeenCalled();
    });

    it('uses the last joined channel if configured to do so', () => {
      const { clientConfiguration, tmiConnection, tmiSession } = setup();
      clientConfiguration.skipAutoRejoin = false;
      tmiSession.lastChannelJoined = '#mikecheb';
      const connectMock = (tmiConnection.commands.connect.execute = jest.fn());

      tmiConnection.tryConnect();

      expect(connectMock.mock.calls[0][0]).toMatchObject({
        joinChannel: '#mikecheb',
      });
    });

    it('skips the last joined channel if configured to do so', () => {
      const { clientConfiguration, tmiConnection, tmiSession } = setup();
      clientConfiguration.skipAutoRejoin = true;
      tmiSession.lastChannelJoined = '#mikecheb';
      const connectMock = (tmiConnection.commands.connect.execute = jest.fn());

      tmiConnection.tryConnect();

      expect(connectMock.mock.calls[0][0]).toMatchObject({
        joinChannel: undefined,
      });
    });

    it('reports connected if already connected', async () => {
      const { tmiConnection } = setup();
      const mockExecute = jest
        .fn()
        .mockResolvedValue({ response: { succeeded: true } });
      tmiConnection.commands.connect.execute = mockExecute;
      tmiConnection.tryConnect();
      // eslint-disable-next-line tachyon/no-mock-clear
      mockExecute.mockClear();
      const { state } = await tmiConnection.tryConnect();
      expect(state).toEqual(TMIConnectionState.Connected);
      expect(mockExecute).not.toHaveBeenCalled();
    });

    it('reports it is connected upon success', async () => {
      const { tmiConnection } = setup();
      tmiConnection.commands.connect.execute = jest
        .fn()
        .mockResolvedValue({ response: { succeeded: true } });
      const { state } = await tmiConnection.tryConnect();
      expect(state).toEqual(TMIConnectionState.Connected);
    });

    it('reports it is disconnected upon failure', async () => {
      const { tmiConnection } = setup();
      tmiConnection.commands.connect.execute = jest
        .fn()
        .mockRejectedValue(new Error('Test connect error'));
      const { state } = await tmiConnection.tryConnect();
      expect(state).toEqual(TMIConnectionState.Disconnected);
    });
  });

  describe('.onReconnect()', () => {
    it('invokes TMIClient.reconnect()', () => {
      const { tmiClient, tmiConnection } = setup();
      tmiClient.reconnect = jest.fn();
      tmiConnection.onReconnect();
      expect(tmiClient.reconnect).toHaveBeenCalled();
    });
  });

  describe('.notifyReconnect()', () => {
    it('emits a reconnecting event', () => {
      const { eventEmitter, tmiConnection } = setup();
      tmiConnection.notifyReconnect('test reconnecting');
      expect(eventEmitter.emit).toHaveBeenCalledWith(
        'reconnecting',
        expect.anything(),
      );
    });
  });

  describe('.notifyReconnected()', () => {
    it('emits a reconnected event', () => {
      const { eventEmitter, tmiConnection } = setup();
      tmiConnection.notifyReconnected(1, 'test reconnected');
      expect(eventEmitter.emit).toHaveBeenCalledWith(
        'reconnected',
        expect.anything(),
      );
    });
  });

  describe('.disconnect()', () => {
    it('closes a connected websocket', async () => {
      const { tmiConnection } = setup();
      tmiConnection.commands.connect.execute = jest
        .fn()
        .mockResolvedValue({ response: { succeeded: true } });
      await tmiConnection.tryConnect();

      const webSocket = (global.WebSocket as any).instances[0];
      tmiConnection.disconnect(false);
      expect(webSocket.close).toHaveBeenCalled();
    });

    it('resets a TMI session', async () => {
      const { tmiConnection, tmiSession } = setup();
      tmiConnection.commands.connect.execute = jest
        .fn()
        .mockResolvedValue({ response: { succeeded: true } });
      await tmiConnection.tryConnect();
      tmiSession.reset = jest.fn();
      tmiConnection.disconnect(false);
      expect(tmiSession.reset).toHaveBeenCalled();
    });

    it('sends a disconnected event', async () => {
      const { eventEmitter, tmiConnection } = setup();
      tmiConnection.commands.connect.execute = jest
        .fn()
        .mockResolvedValue({ response: { succeeded: true } });
      await tmiConnection.tryConnect();
      tmiConnection.disconnect(false);
      expect(eventEmitter.emit).toHaveBeenCalledWith(
        'disconnected',
        expect.anything(),
      );
    });

    it('no-ops if it is not active', () => {
      const { eventEmitter, tmiConnection } = setup();
      tmiConnection.disconnect(false);
      expect(eventEmitter.emit).not.toHaveBeenCalled();
    });

    it('clears session.lastChannelJoined if !shouldReconnect', async () => {
      const { tmiConnection, tmiSession } = setup();
      tmiConnection.commands.connect.execute = jest
        .fn()
        .mockResolvedValue({ response: { succeeded: true } });
      await tmiConnection.tryConnect();
      tmiSession.lastChannelJoined = 'test_channel';
      tmiConnection.disconnect(false);
      expect(tmiSession.lastChannelJoined).toBeFalsy();
    });
  });

  describe('.suppressEvents()', () => {
    it('suppresses subsequent events', async () => {
      const { eventEmitter, tmiConnection } = setup();
      tmiConnection.suppressEvents();
      tmiConnection.notifyReconnect('test reason');
      expect(eventEmitter.emit).not.toHaveBeenCalled();
    });
  });

  describe('.unsuppressEvents()', () => {
    it('emits subsequent events', async () => {
      const { eventEmitter, tmiConnection } = setup();
      tmiConnection.suppressEvents();
      tmiConnection.notifyReconnect('test reason');
      expect(eventEmitter.emit).not.toHaveBeenCalled();
      tmiConnection.unsuppressEvents();
      tmiConnection.notifyReconnect('test reason');
      expect(eventEmitter.emit).toHaveBeenCalled();
    });
  });

  describe('.getCommandTimeout()', () => {
    it('returns at least 600 regardless of latency', () => {
      const { tmiConnection } = setup();
      expect(tmiConnection.getCommandTimeout()).toBeGreaterThanOrEqual(600);
    });
  });

  describe('.isConnected()', () => {
    it('returns true when connected', async () => {
      const { tmiConnection } = setup();
      tmiConnection.commands.connect.execute = jest
        .fn()
        .mockResolvedValue({ response: { succeeded: true } });
      await tmiConnection.tryConnect();
      expect(tmiConnection.isConnected()).toBeTruthy();
    });

    it('returns false when disconnected', async () => {
      const { tmiConnection } = setup();
      expect(tmiConnection.isConnected()).toBeFalsy();
    });
  });

  describe('.pong()', () => {
    it('sends a pong to the websocket', async () => {
      const { tmiConnection } = setup();
      tmiConnection.commands.connect.execute = jest
        .fn()
        .mockResolvedValue({ response: { succeeded: true } });
      await tmiConnection.tryConnect();
      tmiConnection.pong();
      expect((global.WebSocket as any).instances[0].send).toHaveBeenCalled();
    });
  });

  describe('.send()', () => {
    it('sends data to the websocket when connected', async () => {
      const TEST_DATA = 'some data';
      const { tmiConnection } = setup();
      tmiConnection.commands.connect.execute = jest
        .fn()
        .mockResolvedValue({ response: { succeeded: true } });
      await tmiConnection.tryConnect();
      tmiConnection.send(TEST_DATA);
      expect((global.WebSocket as any).instances[0].send).toHaveBeenCalledWith(
        TEST_DATA,
      );
    });
  });

  describe('.injectMessage(data: string)', () => {
    it('emits a message event', () => {
      const { eventEmitter, tmiConnection, tmiParser } = setup();
      tmiParser.msg = jest.fn().mockReturnValue(PARSED_VALID_TMI_MESSAGE);
      tmiConnection.injectMessage(VALID_TMI_MESSAGE);
      expect(eventEmitter.emit).toHaveBeenCalledWith(
        'chat',
        expect.any(Object),
      );
    });
  });
});
