import type EventEmitter from 'eventemitter3';
import type { ClientConfiguration } from './client-configuration';
import type { ConnectCommandRequest } from './commands';
import { EventProcessors } from './event-processors';
import { MessageProcessor } from './message-processor';
import type {
  TMIConnectionResult,
  TMIEventName,
  TMILogger,
  TMISession,
} from './models';
import { TMIConnectionState } from './models';
import type { TMIClient } from './tmi-client';
import { TMICommands } from './tmi-commands';
import type { TMIParser } from './tmi-parser';
import { Utils } from './utils';

export class TMIConnection {
  public readonly logger: TMILogger;
  public readonly commands: TMICommands;

  private readonly eventProcessors: EventProcessors;
  private readonly messageProcessor: MessageProcessor;
  private readonly session: TMISession;
  private readonly configuration: ClientConfiguration;

  private timestampCreated: number;
  private isActive = false;
  private pingLoopHandle = 0;

  private readonly client: TMIClient;
  private ws: WebSocket | null = null;
  private currentLatency = 0;

  constructor(
    client: TMIClient,
    configuration: ClientConfiguration,
    isSilent: boolean,
    emitter: EventEmitter<TMIEventName>,
    session: TMISession,
    parser: TMIParser,
    logger: TMILogger,
  ) {
    this.client = client;
    this.timestampCreated = Date.now();
    this.logger = logger;
    this.session = session;
    this.configuration = configuration;
    this.eventProcessors = new EventProcessors(emitter, isSilent);
    this.commands = new TMICommands(this, session, this.eventProcessors);
    this.messageProcessor = new MessageProcessor(
      parser,
      this.eventProcessors,
      session,
      this,
    );
  }

  public async tryConnect(): Promise<TMIConnectionResult> {
    if (this.isConnected()) {
      this.logger.warn('Attempted to connect, while already connected.');
      return { state: TMIConnectionState.Connected };
    }

    let joinChannel = this.configuration.channel;
    if (!joinChannel && !this.configuration.skipAutoRejoin) {
      joinChannel = this.session.lastChannelJoined;
    }
    this.logger.debug('Connecting...', {
      joinChannel,
      url: this.configuration.connectUrl,
      username: this.configuration.username,
    });
    this.eventProcessors.connecting({
      address: this.configuration.server,
      port: this.configuration.port,
    });

    this.ws = new WebSocket(this.configuration.connectUrl);
    this.ws.onclose = this.onSocketClose;
    this.ws.onmessage = this.onSocketMessage;
    this.ws.onerror = this.onSocketError;

    const connectRequest: ConnectCommandRequest = {
      joinChannel,
      password: this.configuration.password,
      username: this.configuration.username,
      ws: this.ws,
    };

    try {
      const result = await this.commands.connect.execute(
        connectRequest,
        this.configuration.connectTimeout,
      );
      if (result.response.succeeded) {
        this.onConnected();
        return { state: TMIConnectionState.Connected };
      }
    } catch (err) {
      // QOS: Failed to connect to chat.
      this.logger.trackAndWarn(
        'chat_connection_failure',
        {
          reason: Utils.isError(err)
            ? err.toString()
            : 'chat_connection_failure',
          user_id: this.configuration.username,
        },
        'failure connecting to chat',
      );
      this.logger.warn('Failed to connect', err);
    }

    this.onDisconnected('Failed to connect');
    return { state: TMIConnectionState.Disconnected };
  }

  public onReconnect = (): void => {
    this.client.reconnect();
  };

  public notifyReconnect(reason: string): void {
    this.eventProcessors.reconnecting({ reason });
  }

  public notifyReconnected(attempts: number, reason: string): void {
    this.eventProcessors.reconnected({ attempts, reason });
  }

  public disconnect(shouldReconnect: boolean): void {
    if (!this.isActive) {
      return;
    }

    // If we are deliberately disconnecting with no intent to reconnect
    // immediately, clear lastChannelJoined. This is because when connecting again,
    // we want a clear state for tryConnect() above to use instead of
    // our previous state.
    if (!shouldReconnect) {
      this.session.lastChannelJoined = '';
    }

    this.isActive = shouldReconnect;
    this.onDisconnected('User or client initiated');
  }

  public suppressEvents(): void {
    this.eventProcessors.suppress();
  }

  public unsuppressEvents(): void {
    this.eventProcessors.unsuppress();
  }

  public getCommandTimeout(): number {
    return Math.max(this.currentLatency + 100, 600);
  }

  public isConnected(): boolean | null {
    return this.ws && this.ws.readyState === WebSocket.OPEN;
  }

  public pong(): void {
    this.send('PONG');
  }

  public send(data: string): void {
    if (!this.isConnected() || !this.ws) {
      this.logger.warn('Attempted to send data while disconnected', { data });
      return;
    }

    this.ws.send(data);
  }

  public injectMessage(data: string): void {
    this.messageProcessor.processMessage(data);
  }

  private onSocketMessage = (event: MessageEvent) => {
    this.messageProcessor.processMessage(event.data as string);
  };

  private onSocketError = () => {
    // QOS: Failed to connect to chat.
    this.logger.trackAndWarn(
      'chat_connection_failure',
      {
        reason: 'error connecting to websocket',
        user_id: this.configuration.username,
      },
      'failure connecting to chat',
    );
    this.disconnect(true);
  };

  private onSocketClose = (ev: CloseEvent) => {
    if (!this.isActive) {
      this.logger.debug('Socket closed, as expected');
      return;
    }

    // QOS: Socket closed unexpectedly, triggering reconnection.
    // TODO(will): Report a count of socket closures to Spade.
    this.logger.warn('Socket closed unexpectedly', { reason: ev.reason });
    this.onDisconnected('Socket closed unexpectedly');
    this.onReconnect();
  };

  private onConnected(): void {
    this.isActive = true;
    this.pingLoopHandle = window.setInterval(
      this.pingLoop,
      this.configuration.pingInterval,
    );
    this.eventProcessors.connected({
      address: this.configuration.server,
      port: this.configuration.port,
      timestampCreated: this.timestampCreated,
      timestampJoined: Date.now(),
    });
  }

  private onDisconnected(reason: string) {
    if (this.pingLoopHandle) {
      clearInterval(this.pingLoopHandle);
    }
    this.eventProcessors.disconnected({ reason });
    this.session.reset();
    this.shutdownSocket();
  }

  private shutdownSocket = () => {
    try {
      if (this.ws && this.ws.readyState !== WebSocket.CLOSED) {
        this.ws.close();
      }
    } catch (err) {
      // QOS: Failed to close socket on disconnect.
      this.logger.warn('Failed to close socket during disconnect', err);
    }

    this.ws = null;
  };

  private pingLoop = async () => {
    try {
      const result = await this.commands.ping.execute(
        { timestamp: Date.now() },
        this.configuration.pingTimeout,
      );
      this.currentLatency = Date.now() - result.request.timestamp;
    } catch (err) {
      // QOS: PING command failed, triggering reconnection.
      // TODO(will): Report a count of failed PINGs to Spade.
      this.disconnect(true);
    }
  };
}
