import type { TMIMessage } from './models';
import {
  PARSED_TMI_MESSAGE_MULTIPLE_EQUALS_TAG,
  PARSED_VALID_TMI_MESSAGE,
  TMI_MESSAGE_MULTIPLE_EQUALS_TAG,
  VALID_TMI_MESSAGE,
  mockTMILogger,
} from './tests';
import { TMIParser } from './tmi-parser';

const createMessage = (overrides: Partial<TMIMessage> = {}) => ({
  params: [],
  raw: 'Hello there!',
  tags: {},
  ...overrides,
});

const setup = (overrides: Partial<TMIMessage> = {}) => {
  const initialMessage = createMessage(overrides);

  // Provide a shallow copy of the initial message to use for assertions,
  // since the parser mutates the passed message.
  const message = { ...initialMessage };
  const parser = new TMIParser(mockTMILogger());

  return {
    initialMessage,
    message,
    parser,
  };
};

describe('TMIParser', () => {
  it('can be constructed', () => {
    const parser = new TMIParser(mockTMILogger());
    expect(parser).toBeTruthy();
  });

  describe('badges()', () => {
    it("doesn't change the message when the message has no tags", () => {
      const { initialMessage, message, parser } = setup({ tags: {} });

      parser.badges(message);
      expect(message).toEqual(initialMessage);
    });

    it("doesn't change the message when tag.badges are undefined", () => {
      const { initialMessage, message, parser } = setup({
        tags: {
          badges: undefined,
        },
      });

      parser.badges(message);
      expect(message).toEqual(initialMessage);
    });

    it("doesn't change the message when tags.badges is invalid", () => {
      const { initialMessage, message, parser } = setup({
        tags: {
          badges: 'invalidBadge,premium/1',
        },
      });

      parser.badges(message);
      expect(message).toEqual(initialMessage);
    });

    it("doesn't change the message when the message already has parsed badges", () => {
      const { initialMessage, message, parser } = setup({
        badges: {
          foo: 'bar',
        },
        tags: {
          'badge-info': 'broadcaster/foo',
          badges: 'broadcaster/1,premium/1',
        },
      });

      parser.badges(message);
      expect(message).toEqual(initialMessage);
    });

    it('correctly parses badges from the tags', () => {
      const { message, parser } = setup({
        tags: {
          'badge-info': 'subscriber/2',
          badges: 'broadcaster/1,subscriber/1',
        },
      });

      expect(message.badges).toBeUndefined();

      parser.badges(message);
      expect(message.badges).toEqual({
        broadcaster: '1',
        subscriber: '1',
      });
      expect(message.badgeDynamicData).toEqual({
        subscriber: '2',
      });
    });

    it('correctly parses prediction badges from the tags', () => {
      const { message, parser } = setup({
        tags: {
          'badge-info':
            'predictions/test comma ⸝⸝⸝⸝⸝⸝⸝ and // /// //// slash,subscriber/1',
          badges: 'predictions/blue,subscriber/1',
        },
      });

      expect(message.badges).toBeUndefined();

      parser.badges(message);
      expect(message.badges).toEqual({
        predictions: 'blue',
        subscriber: '1',
      });
      expect(message.badgeDynamicData).toEqual({
        predictions: 'test comma ,,,,,,, and // /// //// slash',
        subscriber: '1',
      });
    });
  });

  describe('emotes()', () => {
    it("doesn't change the message when the message already has parsed emotes", () => {
      const { initialMessage, message, parser } = setup({
        emotes: {
          25: {
            id: '25',
            startIndex: 0,
          },
        },
        tags: {
          emotes: '14:0-6',
        },
      });

      parser.emotes(message);
      expect(message).toEqual(initialMessage);
    });

    it("doesn't change the message when the message has no tags", () => {
      const { initialMessage, message, parser } = setup({ tags: {} });

      parser.emotes(message);
      expect(message).toEqual(initialMessage);
    });

    it("doesn't change the message when tags.emotes is undefined", () => {
      const { initialMessage, message, parser } = setup({
        tags: {
          emotes: undefined,
        },
      });

      parser.emotes(message);
      expect(message).toEqual(initialMessage);
    });

    it('correctly parses emotes from the tags', () => {
      const emoteTags = [
        // An emote that is used multiple times throughout the message
        '25:0-4,12-16',
        '36:6-10',
      ];

      const { message, parser } = setup({
        tags: {
          emotes: emoteTags.join('/'),
        },
      });

      expect(message.emotes).toBeUndefined();

      parser.emotes(message);
      expect(message.emotes).toEqual({
        0: {
          id: '25',
          startIndex: 0,
        },
        6: {
          id: '36',
          startIndex: 6,
        },
        12: {
          id: '25',
          startIndex: 12,
        },
      });
    });

    it('skips emotes that are invalid', () => {
      const emoteTags = [
        '25:0-4',
        // Invalid: only has one part
        '1234',
        // Invalid: part is empty string
        '32:',
        '30:7-9',
        // Invalid: indices only has one part
        '32:12',
      ];

      const { message, parser } = setup({
        tags: {
          emotes: emoteTags.join('/'),
        },
      });

      parser.emotes(message);

      // Confirm that the invalid '123' in the tag is skipped.
      expect(message.emotes).toEqual({
        0: {
          id: '25',
          startIndex: 0,
        },
        7: {
          id: '30',
          startIndex: 7,
        },
      });
    });
  });

  describe('msg()', () => {
    it('parses a valid TMI chat message', () => {
      const { parser } = setup();
      const parsedMessage = parser.msg(VALID_TMI_MESSAGE);
      expect(parsedMessage).toEqual(PARSED_VALID_TMI_MESSAGE);
    });

    it('parses a TMI chat message with equals in tag value', () => {
      const { parser } = setup();
      const parsedMessage = parser.msg(TMI_MESSAGE_MULTIPLE_EQUALS_TAG);
      expect(parsedMessage).toEqual(PARSED_TMI_MESSAGE_MULTIPLE_EQUALS_TAG);
    });

    it('returns null for malformed IRC messages', () => {
      const messages = ['@badges=staff/1', '@badges=staff/1  '];
      const { parser } = setup();

      for (const message of messages) {
        expect(parser.msg(message)).toBeNull();
      }
    });

    it('returns null if there is a malformed prefix (nothing after the prefix)', () => {
      const message = ':sender!sender@sender.tmi.twitch.tv';
      const { parser } = setup();
      expect(parser.msg(message)).toBeNull();
    });
  });
});
