import type {
  TMIEmotePosition,
  TMIEmotePositionMap,
  TMILogger,
  TMIMessage,
} from './models';
import { Utils } from './utils';

export class TMIParser {
  private readonly logger: TMILogger;

  constructor(logger: TMILogger) {
    this.logger = logger;
  }

  public badges(message: TMIMessage): void {
    // Already parsed. Do not re-parse.
    if (message.badges) {
      return;
    }

    const rawBadges = message.tags.badges;
    // No source data
    if (!rawBadges) {
      return;
    }

    const badges: Record<string, string> = {};
    const badgesExplode = rawBadges.split(',');

    for (const part of badgesExplode) {
      const parts = part.split('/');
      if (!parts[1]) {
        return;
      }
      badges[parts[0]] = parts[1];
    }

    const rawDynamicData = message.tags['badge-info'];
    const dynamicData: Record<string, string> = {};

    // dynamic badge information for this specific user is available
    if (rawDynamicData) {
      const dynamicDataExplode = rawDynamicData.split(',');
      for (const part of dynamicDataExplode) {
        const indexOfFirstSlash = part.indexOf('/');
        const badgeSetID = part.substring(0, indexOfFirstSlash);
        const badgeTag = part.substring(indexOfFirstSlash + 1);
        if (!badgeTag) {
          continue;
        }
        dynamicData[badgeSetID] = Utils.decodeTag(badgeTag);
      }
    }

    message.badges = badges;
    message.badgeDynamicData = dynamicData;
  }

  // Parse Twitch emotes.
  public emotes(message: TMIMessage): void {
    // Already parsed. Do not re-parse.
    if (message.emotes && Object.keys(message.emotes).length) {
      return;
    }

    const rawEmotes = message.tags.emotes;
    // No source data
    if (!rawEmotes) {
      return;
    }

    const emoticons = rawEmotes.split('/');
    const emotes: TMIEmotePositionMap = {};

    for (const emote of emoticons) {
      const parts = emote.split(':');
      if (parts.length !== 2) {
        this.logger.debug('[Emotes] Skipping invalid emote', { emote, parts });
        continue;
      }

      // the emote id
      const emoteKey = parts[0];
      // comma-delimited start/end-indices
      const emoteValue = parts[1];

      if (!emoteKey || !emoteValue) {
        this.logger.debug('[Emotes] Skipping invalid emote', {
          emote,
          emoteKey,
          emoteValue,
        });
        continue;
      }

      const indexes = emoteValue.split(',');

      for (const index of indexes) {
        const indexSplit = index.split('-');
        if (indexSplit.length !== 2) {
          continue;
        }
        const startIndex = Number(indexSplit[0]);
        const emoteRule: TMIEmotePosition = { id: emoteKey, startIndex };
        emotes[startIndex] = emoteRule;
      }
    }

    message.emotes = emotes;
  }

  // Parse Twitch messages.
  public msg(data: string): TMIMessage | null {
    const message: TMIMessage = {
      params: [],
      raw: data,
      tags: {},
    };

    // Position and nextspace are used by the parser as a reference.
    let position = 0;
    let nextspace = 0;

    // The first thing we check for is IRCv3.2 message tags.
    // http://ircv3.atheme.org/specification/message-tags-3.2
    if (data.charCodeAt(0) === 64) {
      nextspace = data.indexOf(' ');

      // Malformed IRC message.
      if (nextspace === -1) {
        return null;
      }

      // Tags are split by a semi colon.
      const rawTags = data.slice(1, nextspace).split(';');

      for (const tag of rawTags) {
        // Tags delimited by an equals sign are key=value tags.
        // If there's no equals, we assign the tag a value of true.
        const splitPosition = tag.indexOf('=');
        let tagKey, value;
        if (splitPosition !== -1) {
          tagKey = tag.substring(0, splitPosition);
          value = tag.substring(splitPosition + 1);
        } else {
          tagKey = tag;
          value = 'true';
        }

        message.tags[tagKey] = value;
      }

      position = nextspace + 1;
    }

    // Skip any trailing whitespace.
    while (data.charCodeAt(position) === 32) {
      position++;
    }

    // Extract the message's prefix if present. Prefixes are prepended with a colon.
    if (data.charCodeAt(position) === 58) {
      nextspace = data.indexOf(' ', position);

      // If there's nothing after the prefix, deem this message to be malformed.
      if (nextspace === -1) {
        return null;
      }

      message.prefix = data.slice(position + 1, nextspace);
      position = nextspace + 1;

      // Skip any trailing whitespace.
      while (data.charCodeAt(position) === 32) {
        position++;
      }
    }

    nextspace = data.indexOf(' ', position);

    // If there's no more whitespace left, extract everything from the
    // current position to the end of the string as the command.
    if (nextspace === -1) {
      if (data.length > position) {
        message.command = data.slice(position);
        return message;
      }

      return null;
    }

    // Else, the command is the current position up to the next space. After
    // that, we expect some parameters.
    message.command = data.slice(position, nextspace);

    position = nextspace + 1;

    // Skip any trailing whitespace.
    while (data.charCodeAt(position) === 32) {
      position++;
    }

    while (position < data.length) {
      nextspace = data.indexOf(' ', position);

      // If the character is a colon, we've got a trailing parameter.
      // At this point, there are no extra params, so we push everything
      // from after the colon to the end of the string, to the params array
      // and break out of the loop.
      if (data.charCodeAt(position) === 58) {
        message.params.push(data.slice(position + 1));
        break;
      }

      // If we still have some whitespace.
      if (nextspace !== -1) {
        // Push whatever's between the current position and the next
        // space to the params array.
        message.params.push(data.slice(position, nextspace));
        position = nextspace + 1;

        // Skip any trailing whitespace and continue looping.
        while (data.charCodeAt(position) === 32) {
          position++;
        }

        continue;
      }

      // If we don't have any more whitespace and the param isn't trailing,
      // push everything remaining to the params array.
      if (nextspace === -1) {
        message.params.push(data.slice(position));
        break;
      }
    }

    return message;
  }
}
